<?php
/**
 * FZ Customizer API Endpoints
 * Provides REST API endpoints for communication with FZ Customizer app
 */

if (!defined('ABSPATH')) {
    exit;
}

class FZ_Customizer_API {
    
    public function __construct() {
        add_action('rest_api_init', array($this, 'register_routes'));
        
        // Add CORS headers for localhost development
        add_action('rest_api_init', array($this, 'add_cors_headers'), 15);
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Get products endpoint
        register_rest_route('fz-customizer/v1', '/products', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_products'),
            'permission_callback' => array($this, 'check_auth_token'),
        ));
        
        // Get single product endpoint
        register_rest_route('fz-customizer/v1', '/products/(?P<id>\d+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_product'),
            'permission_callback' => array($this, 'check_auth_token'),
        ));
        
        // Verify connection endpoint (public)
        register_rest_route('fz-customizer/v1', '/verify', array(
            'methods' => 'GET',
            'callback' => array($this, 'verify_connection'),
            'permission_callback' => '__return_true',
        ));
        
        // Sync integrations endpoint
        register_rest_route('fz-customizer/v1', '/sync-integrations', array(
            'methods' => 'POST',
            'callback' => array($this, 'sync_integrations'),
            'permission_callback' => array($this, 'check_auth_token'),
        ));
    }
    
    /**
     * Add CORS headers for localhost development
     */
    public function add_cors_headers() {
        // Allow requests from localhost:3000 (Next.js dev server)
        if (isset($_SERVER['HTTP_ORIGIN'])) {
            $allowed_origins = array(
                'http://localhost:3000',
                'http://localhost:3001',
                'https://fzcustomizer.com'
            );
            
            if (in_array($_SERVER['HTTP_ORIGIN'], $allowed_origins)) {
                header("Access-Control-Allow-Origin: " . $_SERVER['HTTP_ORIGIN']);
                header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
                header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Auth-Token");
                header("Access-Control-Allow-Credentials: true");
            }
        }
        
        // Handle preflight requests
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            status_header(200);
            exit();
        }
    }
    
    /**
     * Check auth token from request headers
     */
    public function check_auth_token($request) {
        $auth_token = $request->get_header('X-Auth-Token');

        error_log('[FZ Auth] Header X-Auth-Token: ' . ($auth_token ? $auth_token : 'NOT PROVIDED'));

        if (!$auth_token) {
            // Also check query parameter for easier testing
            $auth_token = $request->get_param('auth_token');
            error_log('[FZ Auth] Query param auth_token: ' . ($auth_token ? $auth_token : 'NOT PROVIDED'));
        }

        if (!$auth_token) {
            error_log('[FZ Auth] ❌ No auth token provided');
            return new WP_Error('no_auth', 'No authentication token provided', array('status' => 401));
        }

        // Get stored auth token from options (check both possible option names)
        $stored_token = get_option('fz_auth_token');
        if (!$stored_token) {
            $stored_token = get_option('fz_customizer_auth_token');
        }

        error_log('[FZ Auth] Stored token: ' . ($stored_token ? $stored_token : 'NONE'));
        error_log('[FZ Auth] Site URL: ' . site_url());

        // Also accept test token for localhost development
        if ($auth_token === 'test-token-123' && strpos(site_url(), 'localhost') !== false) {
            error_log('[FZ Auth] ✅ Test token accepted for localhost');
            return true;
        }

        $is_match = $auth_token === $stored_token;
        error_log('[FZ Auth] Token match: ' . ($is_match ? 'YES' : 'NO'));

        if ($is_match) {
            return true;
        }

        return new WP_Error('invalid_auth', 'Invalid authentication token', array('status' => 401));
    }
    
    /**
     * Get all WooCommerce products
     */
    public function get_products($request) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => -1,
            'post_status' => 'publish'
        );
        
        $products = get_posts($args);
        $formatted_products = array();
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            
            if (!$product) {
                continue;
            }
            
            $image_id = $product->get_image_id();
            $image_url = '';
            
            if ($image_id) {
                $image_url = wp_get_attachment_url($image_id);
            }
            
            $categories = array();
            $terms = get_the_terms($product_post->ID, 'product_cat');
            if ($terms && !is_wp_error($terms)) {
                foreach ($terms as $term) {
                    $categories[] = array(
                        'id' => $term->term_id,
                        'name' => $term->name,
                        'slug' => $term->slug
                    );
                }
            }
            
            $formatted_products[] = array(
                'id' => $product->get_id(),
                'name' => $product->get_name(),
                'description' => $product->get_description(),
                'short_description' => $product->get_short_description(),
                'price' => $product->get_price(),
                'regular_price' => $product->get_regular_price(),
                'sale_price' => $product->get_sale_price(),
                'sku' => $product->get_sku(),
                'stock_quantity' => $product->get_stock_quantity(),
                'stock_status' => $product->get_stock_status(),
                'image_url' => $image_url,
                'categories' => $categories,
                'type' => $product->get_type(),
                'status' => $product->get_status(),
                'permalink' => get_permalink($product_post->ID)
            );
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'products' => $formatted_products,
            'total' => count($formatted_products)
        ), 200);
    }
    
    /**
     * Get single product by ID
     */
    public function get_product($request) {
        $product_id = $request->get_param('id');
        $product = wc_get_product($product_id);
        
        if (!$product) {
            return new WP_REST_Response(array(
                'error' => 'Product not found'
            ), 404);
        }
        
        $image_id = $product->get_image_id();
        $image_url = '';
        
        if ($image_id) {
            $image_url = wp_get_attachment_url($image_id);
        }
        
        $gallery_ids = $product->get_gallery_image_ids();
        $gallery_urls = array();
        
        foreach ($gallery_ids as $gallery_id) {
            $gallery_urls[] = wp_get_attachment_url($gallery_id);
        }
        
        $categories = array();
        $terms = get_the_terms($product_id, 'product_cat');
        if ($terms && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug
                );
            }
        }
        
        $formatted_product = array(
            'id' => $product->get_id(),
            'name' => $product->get_name(),
            'description' => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'price' => $product->get_price(),
            'regular_price' => $product->get_regular_price(),
            'sale_price' => $product->get_sale_price(),
            'sku' => $product->get_sku(),
            'stock_quantity' => $product->get_stock_quantity(),
            'stock_status' => $product->get_stock_status(),
            'image_url' => $image_url,
            'gallery_urls' => $gallery_urls,
            'categories' => $categories,
            'type' => $product->get_type(),
            'status' => $product->get_status(),
            'permalink' => get_permalink($product_id),
            'attributes' => $product->get_attributes(),
            'variations' => $product->is_type('variable') ? $product->get_available_variations() : array()
        );
        
        return new WP_REST_Response(array(
            'success' => true,
            'product' => $formatted_product
        ), 200);
    }
    
    /**
     * Sync integrations from FZ Customizer
     */
    public function sync_integrations($request) {
        $body = $request->get_json_params();
        
        if (empty($body['store_id']) || empty($body['integrations'])) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Missing required data'
            ), 400);
        }
        
        $store_id = sanitize_text_field($body['store_id']);
        $integrations = $body['integrations'];
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'fz_woocommerce_integrations';
        
        $synced = 0;
        $errors = 0;
        
        foreach ($integrations as $integration) {
            if (empty($integration['woocommerce_product_id']) || empty($integration['fz_product_id'])) {
                $errors++;
                continue;
            }
            
            // Check if integration already exists
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT id FROM $table_name 
                 WHERE woocommerce_product_id = %d AND store_id = %s",
                $integration['woocommerce_product_id'],
                $store_id
            ));
            
            if ($existing) {
                // Update existing
                $result = $wpdb->update(
                    $table_name,
                    array('fz_product_id' => $integration['fz_product_id']),
                    array(
                        'woocommerce_product_id' => $integration['woocommerce_product_id'],
                        'store_id' => $store_id
                    ),
                    array('%s'),
                    array('%d', '%s')
                );
            } else {
                // Insert new
                $result = $wpdb->insert(
                    $table_name,
                    array(
                        'store_id' => $store_id,
                        'woocommerce_product_id' => $integration['woocommerce_product_id'],
                        'fz_product_id' => $integration['fz_product_id']
                    ),
                    array('%s', '%d', '%s')
                );
            }
            
            if ($result !== false) {
                $synced++;
            } else {
                $errors++;
                error_log('Failed to sync integration: ' . $wpdb->last_error);
            }
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => "Synced $synced integrations",
            'synced' => $synced,
            'errors' => $errors
        ), 200);
    }
    
    /**
     * Verify connection endpoint
     */
    public function verify_connection($request) {
        $auth_token = $request->get_header('X-Auth-Token');
        
        if (!$auth_token) {
            $auth_token = $request->get_param('auth_token');
        }
        
        $stored_token = get_option('fz_customizer_auth_token');
        $is_authenticated = ($auth_token && $auth_token === $stored_token);
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'FZ Customizer API is working',
            'version' => '1.0.0',
            'woocommerce_version' => WC()->version,
            'site_url' => get_site_url(),
            'authenticated' => $is_authenticated,
            'endpoints' => array(
                'products' => rest_url('fz-customizer/v1/products'),
                'product' => rest_url('fz-customizer/v1/products/{id}'),
                'verify' => rest_url('fz-customizer/v1/verify')
            )
        ), 200);
    }
}

// Initialize the API
new FZ_Customizer_API();