<?php
/**
 * FZ Design Handler - Manages saved designs and integrations with WooCommerce
 */

class FZ_Design_Handler {
    
    public function __construct() {
        // Create database table for designs
        add_action('init', [$this, 'create_designs_table']);
        
        // Handle design save webhook
        add_action('rest_api_init', [$this, 'register_webhook_endpoint']);
        
        // Add design to cart item data
        add_filter('woocommerce_add_cart_item_data', [$this, 'add_design_to_cart'], 10, 3);
        
        // Display design in cart
        add_filter('woocommerce_get_item_data', [$this, 'display_design_in_cart'], 10, 2);
        
        // Add design preview thumbnail to cart item
        add_filter('woocommerce_cart_item_thumbnail', [$this, 'add_design_to_cart_thumbnail'], 10, 3);
        
        // Display design in checkout
        add_filter('woocommerce_checkout_cart_item_quantity', [$this, 'add_design_to_checkout'], 10, 3);
        
        // Save design to order
        add_action('woocommerce_checkout_create_order_line_item', [$this, 'save_design_to_order'], 10, 4);
        
        // Display design in order
        add_action('woocommerce_order_item_meta_end', [$this, 'display_design_in_order'], 10, 3);
        
        // Add design to order emails
        add_action('woocommerce_order_item_meta_start', [$this, 'display_design_in_email'], 10, 3);
        
        // AJAX handler for adding customized product to cart
        add_action('wp_ajax_fz_add_customized_to_cart', [$this, 'ajax_add_to_cart']);
        add_action('wp_ajax_nopriv_fz_add_customized_to_cart', [$this, 'ajax_add_to_cart']);
        
        // AJAX handler for saving design
        add_action('wp_ajax_fz_save_design', [$this, 'ajax_save_design']);
        add_action('wp_ajax_nopriv_fz_save_design', [$this, 'ajax_save_design']);
    }
    
    /**
     * Create database table for storing designs
     */
    public function create_designs_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'fz_saved_designs';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            design_id varchar(255) NOT NULL,
            product_id bigint(20) NOT NULL,
            customer_email varchar(255),
            customer_name varchar(255),
            preview_url text,
            design_data longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY design_id (design_id),
            KEY product_id (product_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Register webhook endpoint for design saves
     */
    public function register_webhook_endpoint() {
        register_rest_route('fz-customizer/v1', '/design-saved', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_design_saved'],
            'permission_callback' => '__return_true', // Public endpoint
        ]);
    }
    
    /**
     * Handle design saved webhook from FZ Customizer
     */
    public function handle_design_saved($request) {
        $params = $request->get_json_params();
        
        if (empty($params['designId']) || empty($params['productId'])) {
            return new WP_Error('missing_params', 'Missing required parameters', ['status' => 400]);
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'fz_saved_designs';
        
        // Get preview URL from multiple possible locations
        $preview_url = $params['previewUrl'] ?? 
                      $params['preview_url'] ?? 
                      $params['preview_image_url'] ?? 
                      $params['customization']['preview_image_url'] ?? 
                      $params['customization']['preview_url'] ?? 
                      '';
        
        // Save design to database
        $wpdb->insert($table_name, [
            'design_id' => sanitize_text_field($params['designId']),
            'product_id' => intval($params['wc_product_id'] ?? $params['productId']),
            'customer_email' => sanitize_email($params['customerEmail'] ?? ''),
            'customer_name' => sanitize_text_field($params['customerName'] ?? ''),
            'preview_url' => esc_url_raw($preview_url),
            'design_data' => wp_json_encode($params['designData'] ?? []),
        ]);
        
        return [
            'success' => true,
            'message' => 'Design saved successfully',
            'design_id' => $params['designId'],
            'preview_url' => $preview_url
        ];
    }
    
    /**
     * Add design to cart item data
     */
    public function add_design_to_cart($cart_item_data, $product_id, $variation_id) {
        if (isset($_REQUEST['fz_design_id'])) {
            $cart_item_data['fz_design_id'] = sanitize_text_field($_REQUEST['fz_design_id']);
            
            // Get design details from database
            global $wpdb;
            $table_name = $wpdb->prefix . 'fz_saved_designs';
            $design = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_name WHERE design_id = %s",
                $cart_item_data['fz_design_id']
            ));
            
            if ($design) {
                $cart_item_data['fz_design_preview'] = $design->preview_url;
                $cart_item_data['fz_design_data'] = $design->design_data;
            }
        }
        
        return $cart_item_data;
    }
    
    /**
     * Display design preview in cart
     */
    public function display_design_in_cart($item_data, $cart_item) {
        if (isset($cart_item['fz_design_id'])) {
            $preview_html = '';
            $download_link = '';
            
            if (!empty($cart_item['fz_design_preview'])) {
                // Create preview thumbnail
                $preview_html = sprintf(
                    '<div style="margin-top: 10px;">
                        <a href="%s" target="_blank" style="display: inline-block;">
                            <img src="%s" style="width: 80px; height: 80px; object-fit: cover; border: 2px solid #ddd; border-radius: 4px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);" alt="Your Custom Design">
                        </a>
                    </div>',
                    esc_url($cart_item['fz_design_preview']),
                    esc_url($cart_item['fz_design_preview'])
                );
                
                // Create download link
                $download_link = sprintf(
                    '<a href="%s" download="custom-design-%s.png" style="color: #2271b1; text-decoration: none; font-size: 12px;">
                        📥 Download Design
                    </a>',
                    esc_url($cart_item['fz_design_preview']),
                    esc_attr($cart_item['fz_design_id'])
                );
            }
            
            // Add custom design info as cart item data
            $item_data[] = [
                'key' => __('🎨 Customization', 'fz-customizer'),
                'value' => sprintf(
                    '<div style="display: flex; align-items: center; gap: 15px;">
                        <span style="color: #155724; font-weight: 500;">✓ Design Applied</span>
                        %s
                    </div>',
                    $download_link
                ),
                'display' => '',
            ];
            
            // Add preview as separate item
            if ($preview_html) {
                $item_data[] = [
                    'key' => '',
                    'value' => $preview_html,
                    'display' => '',
                ];
            }
        }
        
        return $item_data;
    }
    
    /**
     * Add design preview to cart item thumbnail
     */
    public function add_design_to_cart_thumbnail($product_thumbnail, $cart_item, $cart_item_key) {
        if (isset($cart_item['fz_design_preview']) && !empty($cart_item['fz_design_preview'])) {
            // Create composite thumbnail with design overlay
            $design_preview = sprintf(
                '<div style="position: relative; display: inline-block;">
                    %s
                    <div style="position: absolute; bottom: 5px; right: 5px; width: 40px; height: 40px; border: 2px solid white; border-radius: 4px; overflow: hidden; box-shadow: 0 2px 4px rgba(0,0,0,0.2);">
                        <img src="%s" style="width: 100%%; height: 100%%; object-fit: cover;" alt="Custom Design" title="Your Custom Design">
                    </div>
                    <div style="position: absolute; top: 5px; left: 5px; background: #2271b1; color: white; padding: 2px 6px; border-radius: 3px; font-size: 10px; font-weight: bold;">
                        CUSTOMIZED
                    </div>
                </div>',
                $product_thumbnail,
                esc_url($cart_item['fz_design_preview'])
            );
            
            return $design_preview;
        }
        
        return $product_thumbnail;
    }
    
    /**
     * Add design info to checkout page
     */
    public function add_design_to_checkout($quantity_html, $cart_item, $cart_item_key) {
        if (isset($cart_item['fz_design_id'])) {
            $design_info = '<div style="margin-top: 5px; font-size: 12px; color: #2271b1;">';
            $design_info .= '🎨 <strong>Custom Design Applied</strong>';
            
            if (!empty($cart_item['fz_design_preview'])) {
                $design_info .= sprintf(
                    ' <a href="%s" target="_blank" style="color: #2271b1; margin-left: 5px;">View Design</a>',
                    esc_url($cart_item['fz_design_preview'])
                );
            }
            
            $design_info .= '</div>';
            
            return $quantity_html . $design_info;
        }
        
        return $quantity_html;
    }
    
    /**
     * Save design to order item
     */
    public function save_design_to_order($item, $cart_item_key, $values, $order) {
        if (isset($values['fz_design_id'])) {
            $item->add_meta_data('_fz_design_id', $values['fz_design_id']);
            if (isset($values['fz_design_preview'])) {
                $item->add_meta_data('_fz_design_preview', $values['fz_design_preview']);
            }
            if (isset($values['fz_design_data'])) {
                $item->add_meta_data('_fz_design_data', $values['fz_design_data']);
            }
        }
    }
    
    /**
     * Display design in order details
     */
    public function display_design_in_order($item_id, $item, $order) {
        $design_id = $item->get_meta('_fz_design_id');
        $preview_url = $item->get_meta('_fz_design_preview');
        
        if ($design_id) {
            echo '<div style="margin: 10px 0;">';
            echo '<strong>' . __('Custom Design:', 'fz-customizer') . '</strong> ';
            
            if ($preview_url) {
                echo sprintf(
                    '<a href="%s" target="_blank" style="display: inline-block; margin: 5px 0;">
                        <img src="%s" style="width: 100px; height: 100px; object-fit: cover; border: 1px solid #ddd; border-radius: 4px;" alt="Design Preview">
                    </a>
                    <br>
                    <a href="%s" download="design-%s.png" class="button button-small">
                        📥 ' . __('Download Design', 'fz-customizer') . '
                    </a>',
                    esc_url($preview_url),
                    esc_url($preview_url),
                    esc_url($preview_url),
                    esc_attr($design_id)
                );
            } else {
                echo '<span style="color: #2271b1;">✓ ' . __('Customized', 'fz-customizer') . '</span>';
            }
            
            echo '</div>';
        }
    }
    
    /**
     * Display design in order emails
     */
    public function display_design_in_email($item_id, $item, $order) {
        if (!is_wc_endpoint_url() && !is_admin()) { // Only in emails
            $design_id = $item->get_meta('_fz_design_id');
            $preview_url = $item->get_meta('_fz_design_preview');
            
            if ($design_id) {
                echo '<br><strong>' . __('Custom Design:', 'fz-customizer') . '</strong> ';
                
                if ($preview_url) {
                    echo sprintf(
                        '<a href="%s" style="color: #2271b1;">%s</a>',
                        esc_url($preview_url),
                        __('View/Download Design', 'fz-customizer')
                    );
                } else {
                    echo '<span style="color: #2271b1;">✓ ' . __('Customized', 'fz-customizer') . '</span>';
                }
            }
        }
    }
    
    /**
     * AJAX handler for saving design
     */
    public function ajax_save_design() {
        $design_id = isset($_POST['design_id']) ? sanitize_text_field($_POST['design_id']) : '';
        $product_id = isset($_POST['product_id']) ? sanitize_text_field($_POST['product_id']) : '';
        $preview_url = isset($_POST['preview_url']) ? esc_url_raw($_POST['preview_url']) : '';
        $preview_image_url = isset($_POST['preview_image_url']) ? esc_url_raw($_POST['preview_image_url']) : '';
        
        // Use preview_image_url if available, otherwise use preview_url
        $final_preview_url = $preview_image_url ?: $preview_url;
        
        if (!$design_id || !$product_id) {
            wp_send_json_error('Missing required parameters');
        }
        
        // Store design in session for later use
        if (!session_id()) {
            session_start();
        }
        
        $_SESSION['fz_designs'][$product_id] = [
            'design_id' => $design_id,
            'preview_url' => $final_preview_url
        ];
        
        // Also store by WooCommerce product ID if provided
        if (isset($_POST['wc_product_id'])) {
            $wc_product_id = sanitize_text_field($_POST['wc_product_id']);
            $_SESSION['fz_designs']['wc_' . $wc_product_id] = [
                'design_id' => $design_id,
                'preview_url' => $final_preview_url
            ];
        }
        
        wp_send_json_success(['message' => 'Design saved', 'preview_url' => $final_preview_url]);
    }
    
    /**
     * AJAX handler for adding customized product to cart
     */
    public function ajax_add_to_cart() {
        if (!isset($_POST['product_id']) || !isset($_POST['design_id'])) {
            wp_send_json_error('Missing required parameters');
        }
        
        $product_id = sanitize_text_field($_POST['product_id']);
        $design_id = sanitize_text_field($_POST['design_id']);
        $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : 1;
        
        // Get WooCommerce product ID (might be different from FZ product ID)
        global $wpdb;
        $wc_product_id = $wpdb->get_var($wpdb->prepare(
            "SELECT woocommerce_product_id FROM {$wpdb->prefix}fz_woocommerce_integrations 
             WHERE fz_product_id = %s LIMIT 1",
            $product_id
        ));
        
        // If no integration found, try using the ID directly
        if (!$wc_product_id) {
            // Check if it's a numeric WooCommerce product ID
            if (is_numeric($product_id)) {
                $wc_product_id = intval($product_id);
            } else {
                wp_send_json_error(__('Product not found', 'fz-customizer'));
                return;
            }
        } else {
            $wc_product_id = intval($wc_product_id);
        }
        
        // Get preview URL from session if available
        if (!session_id()) {
            session_start();
        }
        
        $preview_url = '';
        // Check multiple possible locations for the preview URL
        if (isset($_SESSION['fz_designs'][$product_id]['preview_url'])) {
            $preview_url = $_SESSION['fz_designs'][$product_id]['preview_url'];
        } elseif (isset($_SESSION['fz_designs']['wc_' . $wc_product_id]['preview_url'])) {
            $preview_url = $_SESSION['fz_designs']['wc_' . $wc_product_id]['preview_url'];
        }
        
        // Also check if passed directly in POST
        if (empty($preview_url) && isset($_POST['preview_url'])) {
            $preview_url = esc_url_raw($_POST['preview_url']);
        }
        
        // Add to cart with design data
        $cart_item_key = WC()->cart->add_to_cart($wc_product_id, $quantity, 0, [], [
            'fz_design_id' => $design_id,
            'fz_design_preview' => $preview_url
        ]);
        
        if ($cart_item_key) {
            wp_send_json_success([
                'message' => __('Product with custom design added to cart!', 'fz-customizer'),
                'cart_url' => wc_get_cart_url()
            ]);
        } else {
            wp_send_json_error(__('Failed to add product to cart', 'fz-customizer'));
        }
    }
}

// Initialize the handler
new FZ_Design_Handler();