<?php
/**
 * Plugin Name: FZ Customizer for WooCommerce - Quick Setup
 * Plugin URI: https://fzcustomizer.com
 * Description: Connect WooCommerce products with FZ Customizer for product personalization
 * Version: 4.6.0
 * Author: FZ Team
 * License: GPL v2 or later
 * Text Domain: fz-customizer
 * WC requires at least: 5.0
 * WC tested up to: 8.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('FZ_CUSTOMIZER_VERSION', '4.6.0');
define('FZ_CUSTOMIZER_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('FZ_CUSTOMIZER_PLUGIN_URL', plugin_dir_url(__FILE__));
define('FZ_DASHBOARD_URL', 'https://fzcustomizer.com');

// Determine if we're running on localhost
$site_url = get_site_url();
$is_localhost = strpos($site_url, 'localhost') !== false || strpos($site_url, '127.0.0.1') !== false;

// Set API URL based on environment
if ($is_localhost) {
    // When running from CLI (cron jobs, WP-CLI, sync scripts, etc) or within Docker
    if (defined('DOING_CRON') || (defined('DOING_AJAX') && DOING_AJAX) || php_sapi_name() === 'cli' || defined('WP_CLI')) {
        // Use host.docker.internal for API calls from within Docker
        define('FZ_API_URL', 'http://host.docker.internal:3000');
    } else {
        // Use localhost for browser-based requests
        define('FZ_API_URL', 'http://localhost:3000');
    }
    // Frontend URL is always localhost for browser access
    define('FZ_FRONTEND_URL', 'http://localhost:3000');
} else {
    // Production environment
    define('FZ_API_URL', 'https://fzcustomizer.com');
    define('FZ_FRONTEND_URL', 'https://fzcustomizer.com');
}

// Include design handler
require_once FZ_CUSTOMIZER_PLUGIN_DIR . 'includes/class-fz-design-handler.php';

// Hook into WooCommerce add to cart to handle design IDs
add_filter('woocommerce_add_cart_item_data', 'fz_add_design_to_cart_item', 10, 3);
function fz_add_design_to_cart_item($cart_item_data, $product_id, $variation_id) {
    // Check for multiple designs first
    if (isset($_POST['fz_designs']) && is_array($_POST['fz_designs'])) {
        // Multiple designs - store them all
        $cart_item_data['fz_designs'] = array_map('sanitize_text_field', $_POST['fz_designs']);
        // Add a unique key to allow this as a single cart item with multiple designs
        $cart_item_data['unique_key'] = md5($product_id . '_designs_' . implode('_', $cart_item_data['fz_designs']) . '_' . time());
    }
    // Check if a single design ID was submitted (backwards compatibility)
    elseif (isset($_POST['fz_design_id']) && !empty($_POST['fz_design_id'])) {
        $design_id = sanitize_text_field($_POST['fz_design_id']);
        $cart_item_data['fz_design_id'] = $design_id;
        // Add a unique key to allow multiple cart items with same product but different designs
        $cart_item_data['unique_key'] = md5($product_id . '_' . $design_id . '_' . time());

        // Also try to get preview URL from session if available
        $preview_key = 'fz_design_preview_' . $product_id;
        if (isset($_SESSION[$preview_key])) {
            $cart_item_data['fz_design_preview'] = $_SESSION[$preview_key];
        }
    }
    // Check GET data (from URL parameters)
    elseif (isset($_GET['fz_design_id']) && !empty($_GET['fz_design_id'])) {
        $design_id = sanitize_text_field($_GET['fz_design_id']);
        $cart_item_data['fz_design_id'] = $design_id;
        $cart_item_data['unique_key'] = md5($product_id . '_' . $design_id . '_' . time());
    }

    return $cart_item_data;
}

// Display design info in cart - simple count
add_filter('woocommerce_get_item_data', 'fz_display_design_in_cart', 10, 2);
function fz_display_design_in_cart($item_data, $cart_item) {
    if (isset($cart_item['fz_designs']) && is_array($cart_item['fz_designs'])) {
        $design_count = count($cart_item['fz_designs']);
        $item_data[] = array(
            'key' => __('Custom Designs', 'fz-customizer'),
            'value' => sprintf('%d design(s)', $design_count)
        );
    } elseif (isset($cart_item['fz_design_id'])) {
        $item_data[] = array(
            'key' => __('Custom Design', 'fz-customizer'),
            'value' => 'Yes'
        );
    }
    return $item_data;
}

// Add view link after product name in cart
add_filter('woocommerce_cart_item_name', 'fz_add_view_link_to_cart_item', 10, 3);
function fz_add_view_link_to_cart_item($product_name, $cart_item, $cart_item_key) {
    // Check if has designs
    if (isset($cart_item['fz_designs']) && is_array($cart_item['fz_designs'])) {
        $first_preview = '';
        foreach ($cart_item['fz_designs'] as $design) {
            if (isset($design['previewUrl'])) {
                $first_preview = $design['previewUrl'];
                break;
            }
        }

        if ($first_preview) {
            $product_name .= sprintf(
                ' <a href="#" class="fz-view-designs" data-preview="%s" style="font-size: 12px; color: #0073aa;">[View Designs]</a>',
                esc_attr($first_preview)
            );
        }
    } elseif (isset($cart_item['fz_design_preview'])) {
        $product_name .= sprintf(
            ' <a href="#" class="fz-view-designs" data-preview="%s" style="font-size: 12px; color: #0073aa;">[View Design]</a>',
            esc_attr($cart_item['fz_design_preview'])
        );
    }

    return $product_name;
}

// Override cart item data display to add button
add_filter('woocommerce_display_item_meta', 'fz_add_view_button_to_meta', 10, 3);
function fz_add_view_button_to_meta($html, $item, $args) {
    // Check if this is cart context
    if (is_cart()) {
        // Check if HTML contains our Custom Design text
        if (strpos($html, 'Custom Design') !== false) {
            // Add JavaScript to inject button after this meta
            $html .= '<script>
                jQuery(function($) {
                    setTimeout(function() {
                        $("dt:contains(\'Custom Design\')").each(function() {
                            var $dd = $(this).next("dd");
                            if ($dd.length && !$dd.find(".fz-view-link").length) {
                                $dd.append(" <a href=\'#\' class=\'fz-view-link\' onclick=\'fzShowDesigns(); return false;\' style=\'color: #0073aa; text-decoration: underline;\'>[View]</a>");
                            }
                        });
                    }, 100);
                });
            </script>';
        }
    }
    return $html;
}

// Add preview button directly via JavaScript
add_action('wp_footer', 'fz_cart_preview_js');
function fz_cart_preview_js() {
    if (!is_cart()) return;
    ?>
    <style>
        .fz-cart-view-btn {
            display: inline-block;
            margin-left: 10px;
            padding: 3px 10px;
            background: #0073aa;
            color: white !important;
            border: none;
            border-radius: 3px;
            font-size: 12px;
            cursor: pointer;
            text-decoration: none !important;
        }
        .fz-cart-view-btn:hover {
            background: #005a87;
            color: white !important;
        }

        /* Modal */
        .fz-preview-modal-cart {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.8);
            z-index: 999999;
            cursor: pointer;
        }
        .fz-preview-modal-cart img {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            max-width: 90%;
            max-height: 90%;
            border-radius: 8px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.5);
        }
        .fz-preview-close {
            position: absolute;
            top: 20px;
            right: 40px;
            color: white;
            font-size: 40px;
            cursor: pointer;
        }
    </style>

    <!-- Modal -->
    <div class="fz-preview-modal-cart" id="fzPreviewModal">
        <span class="fz-preview-close" onclick="document.getElementById('fzPreviewModal').style.display='none'">&times;</span>
        <img id="fzPreviewImage" src="" alt="Design Preview">
    </div>

    <script type="text/javascript">
    jQuery(document).ready(function($) {
        console.log('FZ Cart Preview Script Loaded');

        function addPreviewButtons() {
            console.log('Looking for cart items...');

            // Method 1: For WooCommerce Blocks Cart
            $('.wc-block-components-product-details__value').each(function() {
                const $value = $(this);
                const text = $value.text().trim();

                // Skip if already has button
                if ($value.find('.fz-view-btn').length > 0) {
                    return;
                }

                // Check for design patterns
                if (text === 'Yes' || text.match(/\d+\s+design\(s\)/)) {
                    const $name = $value.prev('.wc-block-components-product-details__name');

                    // Check if this is a Custom Design field
                    if ($name.text().includes('Custom Design')) {
                        console.log('Found Custom Design in Blocks cart:', text);

                        // Add button
                        const buttonHtml = ' <button class="fz-view-btn" style="margin-left: 5px; padding: 2px 8px; background: #0073aa; color: white; border: none; border-radius: 3px; font-size: 11px; cursor: pointer;">View</button>';

                        $value.html(text + buttonHtml);

                        // Attach handler
                        $value.find('.fz-view-btn').on('click', function(e) {
                            e.preventDefault();
                            e.stopPropagation();

                            if (typeof fzShowDesigns !== 'undefined') {
                                fzShowDesigns();
                            } else {
                                // Try to get designs from sessionStorage
                                const allDesigns = JSON.parse(sessionStorage.getItem('fz_all_designs') || '{}');
                                let found = false;

                                for (const productId in allDesigns) {
                                    const designs = allDesigns[productId];
                                    if (designs && designs.length > 0 && designs[0].previewUrl) {
                                        window.open(designs[0].previewUrl, '_blank');
                                        found = true;
                                        break;
                                    }
                                }

                                if (!found) {
                                    alert('Please go to product page to view designs');
                                }
                            }
                        });

                        console.log('Button added to Blocks cart');
                    }
                }
            });

            // Method 2: Classic cart (dd/dt structure)
            $('dd').each(function() {
                const $dd = $(this);

                // Skip if already has button
                if ($dd.find('.fz-view-btn').length > 0) {
                    return;
                }

                const text = $dd.text().trim();
                const $dt = $dd.prev('dt');

                // Check if this is Custom Design field
                if ($dt.length && $dt.text().includes('Custom Design')) {
                    console.log('Found Custom Design in classic cart:', text);

                    if (text.match(/\d+\s+design\(s\)/) || text === 'Yes') {
                        // Add button
                        $dd.html(text + ' <button class="fz-view-btn" style="margin-left: 5px; padding: 2px 8px; background: #0073aa; color: white; border: none; border-radius: 3px; font-size: 11px; cursor: pointer;">View</button>');

                        $dd.find('.fz-view-btn').on('click', function(e) {
                            e.preventDefault();
                            if (typeof fzShowDesigns !== 'undefined') {
                                fzShowDesigns();
                            } else {
                                alert('Opening design preview...');
                            }
                        });

                        console.log('Button added to classic cart');
                    }
                }
            });

            // Method 2: Direct search for variation lists
            $('dl.variation').each(function() {
                const $dl = $(this);
                const text = $dl.text();

                if (text.includes('Custom Design')) {
                    console.log('Found Custom Design in dl.variation');

                    const $row = $dl.closest('tr');
                    const $nameCell = $row.find('td.product-name');

                    if ($nameCell.length && !$nameCell.find('.fz-cart-view-btn').length) {
                        console.log('Adding button via method 2');

                        const btn = '<button class="fz-cart-view-btn" onclick="fzShowDesigns ? fzShowDesigns() : alert(\'Loading...\')" style="margin-left: 10px; padding: 3px 10px; background: #0073aa; color: white; border: none; border-radius: 3px; font-size: 12px; cursor: pointer;">View Design</button>';
                        $nameCell.append(btn);
                    }
                }
            });
        }

        // Run multiple times to ensure execution
        addPreviewButtons();
        setTimeout(addPreviewButtons, 500);
        setTimeout(addPreviewButtons, 1000);
        setTimeout(addPreviewButtons, 2000);

        // Re-run after AJAX updates
        $(document.body).on('updated_wc_div updated_cart_totals', function() {
            console.log('Cart updated, re-adding buttons');
            setTimeout(addPreviewButtons, 100);
        });
    });
    </script>
    <?php
}

// Add JavaScript function for showing designs
add_action('wp_footer', function() {
    ?>
    <script type="text/javascript">
    function fzShowDesigns(designsJson) {
        let designsToShow = [];

        // If designsJson is passed (from cart button), use it
        if (designsJson) {
            try {
                designsToShow = JSON.parse(designsJson);
            } catch (e) {
                console.error('Error parsing designs:', e);
            }
        }

        // Otherwise get from sessionStorage
        if (!designsToShow || designsToShow.length === 0) {
            const allDesigns = JSON.parse(sessionStorage.getItem('fz_all_designs') || '{}');
            // Collect all designs
            for (const productId in allDesigns) {
                const productDesigns = allDesigns[productId];
                if (productDesigns && Array.isArray(productDesigns)) {
                    designsToShow = designsToShow.concat(productDesigns);
                }
            }
        }

        if (designsToShow.length > 0) {
            // Create and show modal
            const modal = document.createElement('div');
            modal.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.8); z-index: 999999; display: flex; align-items: center; justify-content: center;';

            const content = document.createElement('div');
            content.style.cssText = 'background: white; border-radius: 10px; padding: 30px; max-width: 800px; max-height: 80vh; overflow: auto; position: relative;';

            let html = '<button onclick="this.closest(\'div\').parentElement.remove()" style="position: absolute; top: 10px; right: 10px; background: none; border: none; font-size: 24px; cursor: pointer;">&times;</button>';
            html += '<h2 style="margin-top: 0;">Your Custom Designs</h2>';
            html += '<div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; margin-top: 20px;">';

            designsToShow.forEach((design, index) => {
                const qty = design.quantity || 1;
                html += `
                    <div style="border: 1px solid #ddd; border-radius: 8px; padding: 10px; text-align: center; position: relative;">
                        <h4 style="margin: 0 0 10px 0;">Design ${index + 1}</h4>
                        ${qty > 1 ? `<span style="position: absolute; top: 10px; right: 10px; background: #ff6b6b; color: white; padding: 2px 8px; border-radius: 12px; font-size: 12px; font-weight: bold;">×${qty}</span>` : ''}
                        ${design.previewUrl ?
                            `<img src="${design.previewUrl}" style="width: 100%; height: 180px; object-fit: contain; margin-bottom: 10px; cursor: pointer; border: 1px solid #eee; border-radius: 4px;" onclick="window.open('${design.previewUrl}', '_blank')" title="Click to view full size">` :
                            `<div style="width: 100%; height: 180px; background: #f0f0f0; display: flex; align-items: center; justify-content: center; margin-bottom: 10px; border-radius: 4px;">No preview</div>`
                        }
                        ${design.previewUrl ? `<button onclick="window.open('${design.previewUrl}', '_blank')" style="margin-top: 8px; padding: 4px 12px; background: #0073aa; color: white; border: none; border-radius: 3px; cursor: pointer; font-size: 12px;">View Full Size</button>` : ''}
                    </div>
                `;
            });

            html += '</div>';
            content.innerHTML = html;
            modal.appendChild(content);
            document.body.appendChild(modal);

            // Close on overlay click
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    modal.remove();
                }
            });
        } else {
            alert('Design previews not found. Please return to the product page to recreate your designs.');
        }
    }
    </script>
    <?php
});

// Add design ID to order items and send to dashboard
add_action('woocommerce_checkout_create_order_line_item', 'fz_add_design_to_order_items', 10, 4);
function fz_add_design_to_order_items($item, $cart_item_key, $values, $order) {
    // Handle multiple designs
    if (isset($values['fz_designs']) && is_array($values['fz_designs'])) {
        $item->add_meta_data('_fz_designs', json_encode($values['fz_designs']));

        // Store design count for easy access
        $item->add_meta_data('_fz_design_count', count($values['fz_designs']));
    }
    // Handle single design (backwards compatibility)
    elseif (isset($values['fz_design_id'])) {
        $item->add_meta_data('_fz_design_id', $values['fz_design_id']);
        if (isset($values['fz_design_preview'])) {
            $item->add_meta_data('_fz_design_preview', $values['fz_design_preview']);
        }
    }
}

// Send order data to dashboard when order is created
add_action('woocommerce_checkout_order_created', 'fz_send_order_to_dashboard');
function fz_send_order_to_dashboard($order) {
    $store_id = get_option('fz_store_id');
    $auth_token = get_option('fz_auth_token');

    if (!$store_id || !$auth_token) {
        return;
    }

    $designs = array();

    // Collect all designs from order items
    foreach ($order->get_items() as $item_id => $item) {
        $product_id = $item->get_product_id();

        // Check for multiple designs
        $fz_designs = $item->get_meta('_fz_designs');
        if ($fz_designs) {
            $designs_data = json_decode($fz_designs, true);
            if (is_array($designs_data)) {
                foreach ($designs_data as $design) {
                    // Check if design is a string (just the ID) or an object
                    if (is_string($design)) {
                        $designs[] = array(
                            'product_id' => $product_id,
                            'design_id' => $design,
                            'preview_url' => null,
                            'canvas_data' => null,
                            'quantity' => $item->get_quantity()
                        );
                    } else {
                        $designs[] = array(
                            'product_id' => $product_id,
                            'design_id' => $design['id'] ?? null,
                            'preview_url' => $design['previewUrl'] ?? null,
                            'canvas_data' => $design['canvasData'] ?? null,
                            'quantity' => $design['quantity'] ?? 1
                        );
                    }
                }
            }
        }
        // Check for single design
        else {
            $design_id = $item->get_meta('_fz_design_id');
            $preview_url = $item->get_meta('_fz_design_preview');

            if ($design_id) {
                $designs[] = array(
                    'product_id' => $product_id,
                    'design_id' => $design_id,
                    'preview_url' => $preview_url,
                    'quantity' => $item->get_quantity()
                );
            }
        }
    }

    // Only send if we have designs
    if (empty($designs)) {
        return;
    }

    // Prepare order data
    $order_data = array(
        'order_id' => $order->get_id(),
        'order_number' => $order->get_order_number(),
        'status' => $order->get_status(),
        'total' => $order->get_total(),
        'currency' => $order->get_currency(),
        'customer_email' => $order->get_billing_email(),
        'customer_name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
        'designs' => $designs,
        'created_at' => $order->get_date_created()->format('c')
    );

    // Send to dashboard API
    $api_url = defined('FZ_API_URL') ? FZ_API_URL : 'https://fzcustomizer.com';
    $response = wp_remote_post($api_url . '/api/woocommerce/order', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $auth_token,
            'X-Store-ID' => $store_id
        ),
        'body' => json_encode($order_data),
        'timeout' => 10
    ));

    if (is_wp_error($response)) {
        error_log('FZ Customizer: Failed to send order to dashboard - ' . $response->get_error_message());
    }
}

// Update dashboard when order status changes
add_action('woocommerce_order_status_changed', 'fz_update_order_status_in_dashboard', 10, 3);
function fz_update_order_status_in_dashboard($order_id, $old_status, $new_status) {
    $store_id = get_option('fz_store_id');
    $auth_token = get_option('fz_auth_token');

    if (!$store_id || !$auth_token) {
        return;
    }

    $order = wc_get_order($order_id);
    if (!$order) {
        return;
    }

    // Check if order has any customizations
    $has_customizations = false;
    foreach ($order->get_items() as $item) {
        if ($item->get_meta('_fz_designs') || $item->get_meta('_fz_design_id')) {
            $has_customizations = true;
            break;
        }
    }

    if (!$has_customizations) {
        return;
    }

    // Send status update to dashboard
    $api_url = defined('FZ_API_URL') ? FZ_API_URL : 'https://fzcustomizer.com';
    $response = wp_remote_post($api_url . '/api/woocommerce/order-status', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $auth_token,
            'X-Store-ID' => $store_id
        ),
        'body' => json_encode(array(
            'order_id' => $order_id,
            'old_status' => $old_status,
            'new_status' => $new_status,
            'updated_at' => current_time('c')
        )),
        'timeout' => 10
    ));

    if (is_wp_error($response)) {
        error_log('FZ Customizer: Failed to update order status in dashboard - ' . $response->get_error_message());
    }
}

// FZ_IS_LOCALHOST constant
define('FZ_IS_LOCALHOST', $is_localhost);

// Create tables on plugin activation
function fz_create_tables() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    
    // Create integrations table
    $table_name = $wpdb->prefix . 'fz_woocommerce_integrations';
    
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id int(11) NOT NULL AUTO_INCREMENT,
        store_id varchar(255) NOT NULL,
        woocommerce_product_id int(11) NOT NULL,
        fz_product_id varchar(255) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY unique_product (store_id, woocommerce_product_id)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

// Run table creation on activation
register_activation_hook(__FILE__, 'fz_create_tables');

// Also check and create table on admin init (in case it was deleted)
add_action('admin_init', function() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'fz_woocommerce_integrations';
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        fz_create_tables();
    }
});

// Declare HPOS compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// Add admin menu
add_action('admin_menu', function() {
    add_menu_page(
        'FZ Customizer',
        'FZ Customizer',
        'manage_options',
        'fz-customizer',
        'fz_customizer_admin_page',
        'dashicons-art',
        30
    );
});

// Admin page content
function fz_customizer_admin_page() {
    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        echo '<div class="wrap">';
        echo '<h1>FZ Customizer for WooCommerce</h1>';
        echo '<div class="notice notice-error"><p><strong>Error:</strong> WooCommerce is not active. Please install and activate WooCommerce first.</p></div>';
        echo '</div>';
        return;
    }

    // Get saved settings
    $store_id = get_option('fz_store_id', '');
    $auth_token = get_option('fz_auth_token', '');
    $enabled = get_option('fz_customizer_enabled', 'yes');
    $enabled_products = get_option('fz_enabled_products', []);
    $connection_status = get_option('fz_connection_status', '');
    $custom_api_url = get_option('fz_custom_api_url', '');
    $button_text = get_option('fz_button_text', 'Customize This Product');
    
    // Handle form submission
    if (isset($_POST['fz_submit'])) {
        if (wp_verify_nonce($_POST['fz_nonce'], 'fz_settings')) {
            $new_store_id = sanitize_text_field($_POST['fz_store_id']);
            $new_auth_token = sanitize_text_field($_POST['fz_auth_token']);

            update_option('fz_store_id', $new_store_id);
            update_option('fz_auth_token', $new_auth_token);
            update_option('fz_customizer_enabled', sanitize_text_field($_POST['fz_customizer_enabled']));

            // Save custom API URL if provided
            if (isset($_POST['fz_custom_api_url'])) {
                update_option('fz_custom_api_url', sanitize_text_field($_POST['fz_custom_api_url']));
            }

            // Save button text
            if (isset($_POST['fz_button_text'])) {
                update_option('fz_button_text', sanitize_text_field($_POST['fz_button_text']));
            }

            // Save enabled products
            $selected_products = isset($_POST['fz_enabled_products']) ? array_map('intval', $_POST['fz_enabled_products']) : [];
            update_option('fz_enabled_products', $selected_products);

            // Mark as connected if we have both store ID and auth token
            if (!empty($new_store_id) && !empty($new_auth_token)) {
                update_option('fz_connection_status', 'connected');

                // Register connection with dashboard
                $api_url = !empty($custom_api_url) ? $custom_api_url : FZ_API_URL;
                $response = wp_remote_post($api_url . '/api/woocommerce/register-connection', array(
                    'headers' => array('Content-Type' => 'application/json'),
                    'body' => json_encode(array(
                        'store_id' => $new_store_id,
                        'woo_url' => site_url(),
                        'auth_token' => $new_auth_token
                    ))
                ));

                if (!is_wp_error($response)) {
                    $body = wp_remote_retrieve_body($response);
                    error_log('FZ Customizer: Connection registered with dashboard - ' . $body);
                }
            }

            echo '<div class="notice notice-success"><p>✅ Settings saved successfully! Your store is now connected to FZ Customizer.</p></div>';
            
            // Refresh values
            $store_id = get_option('fz_store_id');
            $auth_token = get_option('fz_auth_token');
            $enabled = get_option('fz_customizer_enabled');
            $enabled_products = get_option('fz_enabled_products', []);
            $connection_status = get_option('fz_connection_status', '');
            $custom_api_url = get_option('fz_custom_api_url', '');
            $button_text = get_option('fz_button_text', 'Customize This Product');
        }
    }

    // Get all products (check if WooCommerce is active)
    $products = [];

    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        // WooCommerce not active, show warning later
        $wc_not_active = true;
    } else if (function_exists('wc_get_products')) {
        // Use WooCommerce function if available
        $products = wc_get_products([
            'limit' => -1,
            'status' => 'publish'
        ]);
    } else if (function_exists('wc_get_product')) {
        // Fallback to using WP_Query with wc_get_product
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => -1
        );
        $product_query = new WP_Query($args);
        if ($product_query->have_posts()) {
            while ($product_query->have_posts()) {
                $product_query->the_post();
                $product = wc_get_product(get_the_ID());
                if ($product) {
                    $products[] = $product;
                }
            }
            wp_reset_postdata();
        }
    } else {
        // Final fallback - just get post IDs and basic info
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => -1
        );
        $product_posts = get_posts($args);
        foreach ($product_posts as $post) {
            // Create a simple product object
            $products[] = (object) array(
                'ID' => $post->ID,
                'post_title' => $post->post_title,
                'get_id' => function() use ($post) { return $post->ID; },
                'get_name' => function() use ($post) { return $post->post_title; }
            );
        }
    }
    ?>
    <div class="wrap">
        <h1>🎨 FZ Customizer for WooCommerce</h1>

        <?php if (isset($wc_not_active) && $wc_not_active): ?>
        <div class="notice notice-error">
            <p><strong>Error:</strong> WooCommerce is not active. Please activate WooCommerce to use this plugin.</p>
        </div>
        <?php endif; ?>

        <?php if (empty($store_id) || empty($auth_token)): ?>
        <!-- Quick Setup Section -->
        <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; margin: 20px 0; border-radius: 10px;">
            <h2 style="color: white; margin-top: 0;">⚡ Quick Setup - Connect in 30 Seconds!</h2>
            <p style="font-size: 16px;">Click the button below to automatically connect with FZ Customizer</p>
            
            <button onclick="quickSetup()" style="background: white; color: #667eea; border: none; padding: 15px 30px; font-size: 18px; border-radius: 5px; cursor: pointer; font-weight: bold;">
                🚀 Auto-Connect Now
            </button>
            
            <div id="setup-status" style="margin-top: 20px; display: none;">
                <div class="spinner" style="display: inline-block; margin-right: 10px;"></div>
                <span>Connecting to FZ Customizer...</span>
            </div>
        </div>
        
        <div style="background: #f0f0f1; border-left: 4px solid #2271b1; padding: 15px; margin: 20px 0;">
            <h3 style="margin-top: 0;">Manual Setup (if auto-connect doesn't work)</h3>
            <ol>
                <li>Click "Auto-Connect Now" above</li>
                <li>Login to FZ Dashboard (if not already)</li>
                <li>Copy the Store ID and Auth Token</li>
                <li>Paste them in the form below</li>
            </ol>
        </div>
        <?php else: ?>
        <!-- Connected Status -->
        <div style="background: #d4edda; border: 1px solid #c3e6cb; padding: 20px; margin: 20px 0; border-radius: 5px;">
            <h2 style="color: #155724; margin-top: 0;">✅ Connected to FZ Customizer</h2>
            <p style="color: #155724;">Your store is successfully connected. You can now enable customization for your products below.</p>
            <div style="margin: 15px 0;">
                <button onclick="testConnection()" class="button button-primary">Test Connection</button>
                <button onclick="window.open('<?php echo defined('FZ_FRONTEND_URL') ? FZ_FRONTEND_URL : FZ_DASHBOARD_URL; ?>/dashboard/woocommerce-import?store_id=<?php echo esc_attr($store_id); ?>', '_blank')" class="button">Import Products to Dashboard</button>
                <button onclick="window.open('<?php echo defined('FZ_FRONTEND_URL') ? FZ_FRONTEND_URL : FZ_DASHBOARD_URL; ?>/dashboard', '_blank')" class="button">Open Dashboard</button>
            </div>
            <div id="test-result-connected" style="margin-top: 15px;"></div>
        </div>
        <?php endif; ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('fz_settings', 'fz_nonce'); ?>
            
            <h2>Connection Settings</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="fz_store_id">Store ID</label>
                    </th>
                    <td>
                        <input type="text" 
                               id="fz_store_id" 
                               name="fz_store_id" 
                               value="<?php echo esc_attr($store_id); ?>" 
                               class="regular-text" 
                               placeholder="Will be set by Quick Setup" 
                               style="width: 400px;" />
                        <p class="description">Your unique store identifier from FZ Dashboard</p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="fz_auth_token">Auth Token</label>
                    </th>
                    <td>
                        <input type="text" 
                               id="fz_auth_token" 
                               name="fz_auth_token" 
                               value="<?php echo esc_attr($auth_token); ?>" 
                               class="regular-text" 
                               placeholder="Will be set by Quick Setup"
                               style="width: 400px;" />
                        <p class="description">Authentication token for secure communication</p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="fz_customizer_enabled">Enable Customizer</label>
                    </th>
                    <td>
                        <select id="fz_customizer_enabled" name="fz_customizer_enabled">
                            <option value="yes" <?php selected($enabled, 'yes'); ?>>Yes</option>
                            <option value="no" <?php selected($enabled, 'no'); ?>>No</option>
                        </select>
                        <p class="description">Enable or disable the customization feature globally</p>
                    </td>
                </tr>
                
                <?php if (FZ_IS_LOCALHOST): ?>
                <tr>
                    <th scope="row">
                        <label for="fz_custom_api_url">Custom API URL</label>
                    </th>
                    <td>
                        <input type="text" 
                               id="fz_custom_api_url" 
                               name="fz_custom_api_url" 
                               value="<?php echo esc_attr($custom_api_url); ?>" 
                               class="regular-text" 
                               placeholder="e.g., http://localhost:3000 or http://localhost:3001"
                               style="width: 400px;" />
                        <p class="description">
                            For development only. Leave empty to auto-detect. Current: <strong><?php echo FZ_API_URL; ?></strong>
                        </p>
                    </td>
                </tr>
                <?php endif; ?>
                
                <tr>
                    <th scope="row">
                        <label for="fz_button_text">Button Text</label>
                    </th>
                    <td>
                        <input type="text" 
                               id="fz_button_text" 
                               name="fz_button_text" 
                               value="<?php echo esc_attr($button_text); ?>" 
                               class="regular-text" 
                               placeholder="Customize This Product"
                               style="width: 400px;" />
                        <p class="description">Text to display on the customize button</p>
                    </td>
                </tr>
            </table>
            
            <h2>Product Selection</h2>
            <p>Choose which products should have the "Customize" button:</p>
            
            <div style="background: white; border: 1px solid #c3c4c7; padding: 15px; margin: 10px 0;">
                <div style="margin-bottom: 10px;">
                    <button type="button" class="button" onclick="selectAll()">Select All</button>
                    <button type="button" class="button" onclick="selectNone()">Select None</button>
                    <button type="button" class="button" onclick="selectByCategory()">Select by Category</button>
                </div>
                
                <div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; background: #f9f9f9;">
                    <?php foreach ($products as $product): ?>
                        <label style="display: block; margin: 8px 0; padding: 8px; background: white; border-radius: 3px;">
                            <input type="checkbox" 
                                   name="fz_enabled_products[]" 
                                   value="<?php echo $product->get_id(); ?>"
                                   <?php checked(in_array($product->get_id(), $enabled_products)); ?> />
                            <strong><?php echo esc_html($product->get_name()); ?></strong>
                            <span style="color: #666;">(ID: <?php echo $product->get_id(); ?>)</span>
                            <?php if ($product->get_price()): ?>
                                <span style="color: #2271b1;">- <?php echo wc_price($product->get_price()); ?></span>
                            <?php endif; ?>
                        </label>
                    <?php endforeach; ?>
                </div>
                
                <p class="description" style="margin-top: 10px;">
                    💡 Tip: Only select products that can be customized (e.g., T-shirts, mugs, prints)
                </p>
            </div>
            
            <p class="submit">
                <input type="submit" name="fz_submit" class="button-primary" value="Save Settings" />
            </p>
        </form>
        
        <div id="test-result"></div>
        
        <script>
        let messageHandlerAdded = false;
        
        async function quickSetup() {
            const statusDiv = document.getElementById('setup-status');
            statusDiv.style.display = 'block';

            try {
                // Open setup window with parameters
                const setupUrl = '<?php echo FZ_API_URL; ?>/dashboard/woocommerce-setup?from_woo=1&woo_url=' +
                                encodeURIComponent('<?php echo site_url(); ?>');
                console.log('Opening setup URL:', setupUrl);
                const setupWindow = window.open(setupUrl, 'fz-setup', 'width=800,height=700');
                
                // Add message handler only once
                if (!messageHandlerAdded) {
                    messageHandlerAdded = true;
                    
                    window.addEventListener('message', async function(event) {
                        console.log('Received message from:', event.origin);
                        console.log('Message data:', event.data);
                        
                        // Security check - allow localhost for testing
                        if (!event.origin.includes('fz-customizer') && !event.origin.includes('localhost:3000')) {
                            console.log('Origin not allowed:', event.origin);
                            return;
                        }
                        
                        if (event.data && event.data.type === 'fz-setup-complete') {
                            console.log('Setup complete! Received data:', event.data);
                            
                            // Auto-fill the form with received data
                            const storeIdField = document.getElementById('fz_store_id');
                            const authTokenField = document.getElementById('fz_auth_token');
                            
                            if (storeIdField && event.data.store_id) {
                                storeIdField.value = event.data.store_id;
                                console.log('Store ID set to:', event.data.store_id);
                            }
                            
                            if (authTokenField && event.data.auth_token) {
                                authTokenField.value = event.data.auth_token;
                                console.log('Auth Token set successfully');
                            }
                            
                            statusDiv.innerHTML = '<div style="color: #4ade80;">✅ Connection successful! Store ID: ' + (event.data.store_id || 'Not received') + '</div>';
                            
                            // Close setup window
                            if (setupWindow && !setupWindow.closed) {
                                setupWindow.close();
                            }
                            
                            // Auto-save if both fields are filled
                            if (storeIdField.value && authTokenField.value) {
                                console.log('Both fields filled. Store ID:', storeIdField.value, 'Auth Token:', authTokenField.value.substring(0, 10) + '...');

                                // Register connection with dashboard first
                                fetch('<?php echo FZ_API_URL; ?>/api/woocommerce/register-connection', {
                                    method: 'POST',
                                    headers: {
                                        'Content-Type': 'application/json'
                                    },
                                    body: JSON.stringify({
                                        store_id: storeIdField.value,
                                        woo_url: '<?php echo site_url(); ?>',
                                        auth_token: authTokenField.value
                                    })
                                }).then(response => response.json())
                                .then(data => {
                                    console.log('Connection registered with dashboard:', data);
                                }).catch(error => {
                                    console.error('Failed to register connection:', error);
                                });

                                setTimeout(() => {
                                    if (confirm('Connection successful! Save settings now?')) {
                                        // Log form data before submission
                                        console.log('Submitting form with Store ID:', storeIdField.value);

                                        // Create a hidden input to simulate the submit button
                                        const form = document.querySelector('form');
                                        const hiddenSubmit = document.createElement('input');
                                        hiddenSubmit.type = 'hidden';
                                        hiddenSubmit.name = 'fz_submit';
                                        hiddenSubmit.value = 'Save Settings';
                                        form.appendChild(hiddenSubmit);

                                        // Double-check the values are still in the fields
                                        console.log('Final check - Store ID field value:', document.getElementById('fz_store_id').value);
                                        console.log('Final check - Auth Token field value:', document.getElementById('fz_auth_token').value ? 'Present' : 'Missing');

                                        form.submit();
                                    }
                                }, 500);
                            } else {
                                console.error('Fields not filled. Store ID:', storeIdField.value, 'Auth Token:', authTokenField.value);
                                alert('Please check that both Store ID and Auth Token are filled, then save settings manually.');
                            }
                        }
                    });
                }
                
            } catch (error) {
                console.error('Setup error:', error);
                statusDiv.innerHTML = '<div style="color: #ef4444;">❌ Setup failed. Error: ' + error.message + '</div>';
            }
        }
        
        async function testConnection() {
            // Try to find the result div - use connected version if available, otherwise use the one below form
            const resultDiv = document.getElementById('test-result-connected') || document.getElementById('test-result');
            if (!resultDiv) {
                alert('Error: Could not find result container');
                return;
            }

            const storeId = document.getElementById('fz_store_id').value;
            const authToken = document.getElementById('fz_auth_token').value;

            if (!storeId || !authToken) {
                resultDiv.innerHTML = '<div class="notice notice-error"><p>❌ Please enter Store ID and Auth Token</p></div>';
                return;
            }

            resultDiv.innerHTML = '<div class="notice notice-info"><p>⏳ Testing connection...</p></div>';

            try {
                const response = await fetch('<?php echo FZ_API_URL; ?>/api/woocommerce/test-connection?' + new URLSearchParams({
                    store_id: storeId
                }), {
                    method: 'GET',
                    headers: {
                        'Authorization': 'Bearer ' + authToken,
                        'Content-Type': 'application/json'
                    }
                });

                const data = await response.json();
                console.log('Test connection response:', data);

                if (data.status === 'ok') {
                    resultDiv.innerHTML = '<div class="notice notice-success"><p>✅ Connection successful! Store: <strong>' + data.store_name + '</strong></p></div>';

                    // Also import products to show connection is working
                    importProducts();
                } else {
                    resultDiv.innerHTML = '<div class="notice notice-error"><p>❌ Connection failed: ' + (data.error || 'Unknown error') + '</p></div>';
                }
            } catch (error) {
                console.error('Connection test error:', error);
                resultDiv.innerHTML = '<div class="notice notice-error"><p>❌ Connection error: ' + error.message + '</p></div>';
            }
        }

        async function importProducts() {
            const storeId = document.getElementById('fz_store_id').value;
            const authToken = document.getElementById('fz_auth_token').value;

            if (!storeId || !authToken) {
                console.log('Missing store ID or auth token for import');
                return;
            }

            try {
                const response = await fetch('<?php echo FZ_API_URL; ?>/api/woocommerce/import-products?' + new URLSearchParams({
                    store_id: storeId,
                    woo_url: '<?php echo site_url(); ?>'
                }), {
                    method: 'GET',
                    headers: {
                        'Authorization': 'Bearer ' + authToken,
                        'Content-Type': 'application/json'
                    }
                });

                if (response.ok) {
                    const data = await response.json();
                    console.log('Products imported:', data);
                } else {
                    console.error('Import failed:', await response.text());
                }
            } catch (error) {
                console.error('Import error:', error);
            }
        }
        
        function selectAll() {
            document.querySelectorAll('input[name="fz_enabled_products[]"]').forEach(cb => cb.checked = true);
        }
        
        function selectNone() {
            document.querySelectorAll('input[name="fz_enabled_products[]"]').forEach(cb => cb.checked = false);
        }
        
        function selectByCategory() {
            alert('Select products by category - coming soon!');
        }
        </script>
        
        <hr />
        
        <h2>📊 Status</h2>
        <table class="widefat" style="max-width: 600px;">
            <tr>
                <td><strong>Plugin Version:</strong></td>
                <td><?php echo FZ_CUSTOMIZER_VERSION; ?></td>
            </tr>
            <tr>
                <td><strong>Connection Status:</strong></td>
                <td><?php echo ($store_id && $auth_token) ? '✅ Connected' : '❌ Not Connected'; ?></td>
            </tr>
            <tr>
                <td><strong>Enabled Products:</strong></td>
                <td><?php echo count($enabled_products); ?> of <?php echo count($products); ?> products</td>
            </tr>
            <tr>
                <td><strong>Dashboard URL:</strong></td>
                <td><a href="<?php echo FZ_DASHBOARD_URL; ?>" target="_blank"><?php echo FZ_DASHBOARD_URL; ?></a></td>
            </tr>
        </table>
        
        <hr />
        
        <h2>🛍️ Test Products</h2>
        <p>If the shop page is not working, you can test with these products:</p>
        <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; margin: 20px 0;">
            <?php
            $enabled_products = get_option('fz_enabled_products', []);

            // Get test products with comprehensive fallback
            $test_products = [];
            if (!class_exists('WooCommerce')) {
                // WooCommerce not active
                $test_products = [];
            } else if (function_exists('wc_get_products')) {
                $test_products = wc_get_products(['limit' => 6, 'status' => 'publish']);
            } else if (function_exists('wc_get_product')) {
                // Fallback to WP_Query with wc_get_product
                $args = array(
                    'post_type' => 'product',
                    'post_status' => 'publish',
                    'posts_per_page' => 6
                );
                $product_query = new WP_Query($args);
                if ($product_query->have_posts()) {
                    while ($product_query->have_posts()) {
                        $product_query->the_post();
                        $product = wc_get_product(get_the_ID());
                        if ($product) {
                            $test_products[] = $product;
                        }
                    }
                    wp_reset_postdata();
                }
            } else {
                // Final fallback - basic product info
                $args = array(
                    'post_type' => 'product',
                    'post_status' => 'publish',
                    'posts_per_page' => 6
                );
                $product_posts = get_posts($args);
                foreach ($product_posts as $post) {
                    $test_products[] = (object) array(
                        'ID' => $post->ID,
                        'post_title' => $post->post_title,
                        'get_id' => function() use ($post) { return $post->ID; },
                        'get_name' => function() use ($post) { return $post->post_title; },
                        'get_price' => function() use ($post) { return get_post_meta($post->ID, '_price', true); }
                    );
                }
            }

            foreach ($test_products as $product): 
                if (!in_array($product->get_id(), $enabled_products)) continue;
            ?>
                <div style="border: 1px solid #ddd; padding: 15px; border-radius: 5px; text-align: center;">
                    <h4><?php echo esc_html($product->get_name()); ?></h4>
                    <p><?php
                        $price = is_callable([$product, 'get_price']) ? $product->get_price() : '';
                        echo function_exists('wc_price') ? wc_price($price) : '$' . $price;
                    ?></p>
                    <?php 
                    $store_id = get_option('fz_store_id');
                    if ($store_id):
                        $base_url = defined('FZ_FRONTEND_URL') ? FZ_FRONTEND_URL : (FZ_IS_LOCALHOST ? 'http://localhost:3000' : FZ_DASHBOARD_URL);
                        $customize_url = $base_url . '/editor?productId=' . $product->get_id() . '&woocommerce=1&store_id=' . $store_id;
                    ?>
                        <a href="<?php echo esc_url($customize_url); ?>" 
                           target="_blank" 
                           style="display: inline-block; margin-top: 10px; padding: 10px 20px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; text-decoration: none; border-radius: 5px;">
                            🎨 Customize
                        </a>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>
        <?php if (empty($enabled_products)): ?>
            <p style="color: #e74c3c;">⚠️ No products selected for customization. Please select products above and save settings.</p>
        <?php endif; ?>
    </div>
    <?php
}

// Add customize button ONLY to single product page (not in product lists)
// add_action('woocommerce_after_shop_loop_item', 'fz_add_customize_button_loop', 15); // Commented out - no button in lists
add_action('woocommerce_single_product_summary', 'fz_add_customize_button_single', 35);

function fz_add_customize_button_loop() {
    fz_add_customize_button('loop');
}

function fz_add_customize_button_single() {
    fz_add_customize_button('single');
}

function fz_add_customize_button($context = 'loop') {
    if (get_option('fz_customizer_enabled') !== 'yes') {
        return;
    }
    
    global $product;
    $enabled_products = get_option('fz_enabled_products', []);
    
    // Only show button for selected products
    if (!in_array($product->get_id(), $enabled_products)) {
        return;
    }
    
    $store_id = get_option('fz_store_id');
    if (!$store_id) {
        return;
    }
    
    // Get the FZ product ID from integrations table
    global $wpdb;
    $integration = $wpdb->get_row($wpdb->prepare(
        "SELECT fz_product_id FROM {$wpdb->prefix}fz_woocommerce_integrations 
         WHERE woocommerce_product_id = %d AND store_id = %s",
        $product->get_id(),
        $store_id
    ));
    
    // If no integration found, try using the WooCommerce product ID directly
    $fz_product_id = $integration ? $integration->fz_product_id : null;
    
    // Build customize URL - use appropriate domain
    // Use the editor page for product customization with all features
    $base_url = defined('FZ_FRONTEND_URL') ? FZ_FRONTEND_URL : (FZ_IS_LOCALHOST ? 'http://localhost:3000' : FZ_DASHBOARD_URL);
    $product_id_to_use = $fz_product_id ?: $product->get_id();
    $customize_url = $base_url . '/editor?productId=' . $product_id_to_use .
                     '&woocommerce=1&store_id=' . $store_id .
                     '&wc_product_id=' . $product->get_id() .
                     '&wc_site_url=' . urlencode(site_url()) .
                     '&auth_token=' . urlencode(get_option('fz_auth_token', ''));
    
    $button_class = ($context === 'single') ? 'single_add_to_cart_button button alt' : 'button';
    $button_text = get_option('fz_button_text', 'Customize This Product');
    
    // Simple container for customization
    ?>
    <div class="fz-customizer-section" data-product-id="<?php echo esc_attr($product->get_id()); ?>" data-fz-product-id="<?php echo esc_attr($product_id_to_use); ?>">
        <button type="button"
           class="<?php echo esc_attr($button_class); ?> fz-customize-button"
           data-customize-url="<?php echo esc_url($customize_url); ?>"
           data-product-id="<?php echo esc_attr($product->get_id()); ?>"
           data-wc-product-id="<?php echo esc_attr($product->get_id()); ?>"
           data-product-name="<?php echo esc_attr($product->get_name()); ?>"
           style="margin: 5px 0;">
            🎨 <?php echo esc_html($button_text); ?>
        </button>
        <!-- Container for multi-design manager - handles all design display -->
        <div id="fz-designs-container" class="fz-designs-container"></div>
    </div>
    <?php
}

// Enqueue multi-design manager scripts and styles
add_action('wp_enqueue_scripts', function() {
    if (is_product()) {
        // Enqueue multi-design CSS
        wp_enqueue_style(
            'fz-multi-design',
            FZ_CUSTOMIZER_PLUGIN_URL . 'assets/css/multi-design.css',
            [],
            FZ_CUSTOMIZER_VERSION
        );

        // Enqueue multi-design JS with jQuery dependency
        wp_enqueue_script(
            'fz-multi-design-manager',
            FZ_CUSTOMIZER_PLUGIN_URL . 'assets/js/multi-design-manager.js',
            ['jquery'],
            FZ_CUSTOMIZER_VERSION,
            true
        );

        // Localize script with necessary data
        wp_localize_script('fz-multi-design-manager', 'fz_customizer_params', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('fz_customizer_nonce'),
            'product_id' => get_the_ID()
        ]);
    }
});

// Add custom CSS and JavaScript for modal
add_action('wp_head', function() {
    ?>
    <script>
        // Make AJAX URL and nonce available
        window.fz_ajax_url = '<?php echo admin_url('admin-ajax.php'); ?>';
        window.fz_nonce = '<?php echo wp_create_nonce('fz_customizer_nonce'); ?>';
    </script>
    <style>
        /* Let WooCommerce theme styles apply, only add minor enhancements */
        .fz-customize-button {
            transition: all 0.2s ease;
            cursor: pointer;
        }
        .fz-customize-button:hover {
            transform: scale(1.02);
        }
        
        /* Modal styles */
        .fz-customize-modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            z-index: 999999;
            animation: fadeIn 0.3s;
        }
        
        .fz-customize-modal.active {
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .fz-customize-modal-content {
            position: relative;
            width: 100%;
            height: 100%;
            max-width: 100%;
            background: white;
            border-radius: 0;
            overflow: hidden;
            animation: slideIn 0.3s;
        }
        
        .fz-customize-modal-header {
            /* Hidden - header is now inside iframe */
            display: none;
        }
        
        .fz-customize-modal-close {
            background: transparent;
            border: 1px solid #e0e0e0;
            color: #666;
            font-size: 20px;
            cursor: pointer;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
        }
        
        .fz-customize-modal-close:hover {
            background: #f5f5f5;
            color: #333;
            border-color: #999;
        }

        .fz-customize-iframe {
            width: 100%;
            height: 100%;
            border: none;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        @keyframes slideIn {
            from { transform: translateY(-20px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }
    </style>
    
    <!-- Modal HTML -->
    <div id="fz-customize-modal" class="fz-customize-modal">
        <div class="fz-customize-modal-content">
            <div class="fz-customize-modal-header">
                <h3 id="fz-modal-title">Customize Your Product</h3>
                <button type="button" class="fz-customize-modal-close" onclick="closeFZModal()">&times;</button>
            </div>
            <iframe id="fz-customize-iframe" class="fz-customize-iframe" src="about:blank"></iframe>
        </div>
    </div>
    
    <script>
        function openFZModal(url, productName) {
            const modal = document.getElementById('fz-customize-modal');
            const iframe = document.getElementById('fz-customize-iframe');
            const title = document.getElementById('fz-modal-title');

            // Parse the URL to check for existing savedDesignId
            const urlParams = new URL(url, window.location.origin);
            const productId = urlParams.searchParams.get('productId');
            const existingSavedDesignId = urlParams.searchParams.get('savedDesignId');

            // Only auto-load a design if no specific design is requested
            if (productId && !existingSavedDesignId) {
                // Check sessionStorage first, then cookie
                let savedDesignData = sessionStorage.getItem('fz_design_' + productId);
                let savedDesignId = null;

                if (savedDesignData) {
                    try {
                        // Parse the JSON to get just the design ID
                        const designData = JSON.parse(savedDesignData);
                        savedDesignId = designData.designId;
                        console.log('Found saved design in sessionStorage:', designData, 'for product:', productId);
                    } catch (e) {
                        // If it's not JSON, treat it as the ID directly
                        savedDesignId = savedDesignData;
                    }
                }

                if (!savedDesignId) {
                    // Check cookie as fallback
                    const cookieName = 'fz_design_' + productId + '=';
                    const cookies = document.cookie.split(';');
                    for (let cookie of cookies) {
                        cookie = cookie.trim();
                        if (cookie.indexOf(cookieName) === 0) {
                            const cookieValue = cookie.substring(cookieName.length);
                            try {
                                const designData = JSON.parse(decodeURIComponent(cookieValue));
                                savedDesignId = designData.designId;
                            } catch (e) {
                                savedDesignId = cookieValue;
                            }
                            break;
                        }
                    }
                }
                
                if (savedDesignId) {
                    console.log('Adding savedDesignId to URL:', savedDesignId);
                    // Add savedDesignId to URL
                    urlParams.searchParams.set('savedDesignId', savedDesignId);
                    url = urlParams.toString();
                }
            }
            
            title.textContent = 'Customize: ' + productName;
            iframe.src = url;
            modal.classList.add('active');
            
            // Prevent body scroll when modal is open
            document.body.style.overflow = 'hidden';
        }
        
        function closeFZModal() {
            const modal = document.getElementById('fz-customize-modal');
            const iframe = document.getElementById('fz-customize-iframe');
            
            modal.classList.remove('active');
            iframe.src = 'about:blank';
            
            // Restore body scroll
            document.body.style.overflow = '';
        }
        
        // Close modal on escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeFZModal();
            }
        });

        // Function to show designs modal from data (for cart page)
        function showCartDesignsModalFromData(designs) {
            // Create modal to show designs
            const modal = document.createElement('div');
            modal.className = 'fz-cart-designs-modal';
            modal.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.8); z-index: 999999; display: flex; align-items: center; justify-content: center;';

            const content = document.createElement('div');
            content.style.cssText = 'background: white; border-radius: 10px; padding: 30px; max-width: 800px; max-height: 80vh; overflow: auto; position: relative;';

            let html = '<button onclick="this.closest(\'.fz-cart-designs-modal\').remove()" style="position: absolute; top: 10px; right: 10px; background: none; border: none; font-size: 24px; cursor: pointer;">&times;</button>';
            html += '<h2 style="margin-top: 0;">Your Custom Designs</h2>';

            if (designs.length > 0) {
                html += '<div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; margin-top: 20px;">';
                designs.forEach((design, index) => {
                    const qty = design.quantity || 1;
                    html += `
                        <div style="border: 1px solid #ddd; border-radius: 8px; padding: 10px; text-align: center; position: relative;">
                            <h4 style="margin: 0 0 10px 0;">Design ${index + 1}</h4>
                            ${qty > 1 ? `<span style="position: absolute; top: 10px; right: 10px; background: #ff6b6b; color: white; padding: 2px 8px; border-radius: 12px; font-size: 12px; font-weight: bold;">×${qty}</span>` : ''}
                            ${design.previewUrl ?
                                `<img src="${design.previewUrl}" style="width: 100%; height: 180px; object-fit: contain; margin-bottom: 10px; cursor: pointer; border: 1px solid #eee; border-radius: 4px;" onclick="window.open('${design.previewUrl}', '_blank')" title="Click to view full size">` :
                                `<div style="width: 100%; height: 180px; background: #f0f0f0; display: flex; align-items: center; justify-content: center; margin-bottom: 10px; border-radius: 4px;">No preview</div>`
                            }
                            <div style="margin-top: 10px;">
                                <small style="color: #666; display: block;">ID: ${(design.designId || design.id || '').substring(0, 8)}...</small>
                                ${design.previewUrl ? `<button onclick="window.open('${design.previewUrl}', '_blank')" style="margin-top: 8px; padding: 4px 12px; background: #0073aa; color: white; border: none; border-radius: 3px; cursor: pointer; font-size: 12px;">View Full Size</button>` : ''}
                            </div>
                        </div>
                    `;
                });
                html += '</div>';
            } else {
                html += '<p>No designs found. Please make sure your designs are saved.</p>';
            }

            content.innerHTML = html;
            modal.appendChild(content);
            document.body.appendChild(modal);

            // Close on overlay click
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    modal.remove();
                }
            });
        }

        // Function to show designs modal from cart
        function showCartDesignsModal(designIds) {
            // Get all designs from sessionStorage
            const allDesigns = JSON.parse(sessionStorage.getItem('fz_all_designs') || '{}');
            let foundDesigns = [];

            // Search for designs across all products
            for (const productId in allDesigns) {
                const productDesigns = allDesigns[productId];
                if (productDesigns && Array.isArray(productDesigns)) {
                    productDesigns.forEach(design => {
                        if (designIds.includes(design.designId)) {
                            foundDesigns.push(design);
                        }
                    });
                }
            }

            // Create modal to show designs
            const modal = document.createElement('div');
            modal.className = 'fz-cart-designs-modal';
            modal.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.8); z-index: 999999; display: flex; align-items: center; justify-content: center;';

            const content = document.createElement('div');
            content.style.cssText = 'background: white; border-radius: 10px; padding: 30px; max-width: 800px; max-height: 80vh; overflow: auto; position: relative;';

            let html = '<button onclick="this.closest(\'.fz-cart-designs-modal\').remove()" style="position: absolute; top: 10px; right: 10px; background: none; border: none; font-size: 24px; cursor: pointer;">&times;</button>';
            html += '<h2 style="margin-top: 0;">Your Custom Designs</h2>';

            if (foundDesigns.length > 0) {
                html += '<div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 20px; margin-top: 20px;">';
                foundDesigns.forEach((design, index) => {
                    const qty = design.quantity || 1;
                    html += `
                        <div style="border: 1px solid #ddd; border-radius: 8px; padding: 10px; text-align: center; position: relative;">
                            <h4 style="margin: 0 0 10px 0;">Design ${index + 1}</h4>
                            ${qty > 1 ? `<span style="position: absolute; top: 10px; right: 10px; background: #ff6b6b; color: white; padding: 2px 8px; border-radius: 12px; font-size: 12px; font-weight: bold;">×${qty}</span>` : ''}
                            ${design.previewUrl ?
                                `<img src="${design.previewUrl}" style="width: 100%; height: 180px; object-fit: contain; margin-bottom: 10px; cursor: pointer; border: 1px solid #eee; border-radius: 4px;" onclick="window.open('${design.previewUrl}', '_blank')" title="Click to view full size">` :
                                `<div style="width: 100%; height: 180px; background: #f0f0f0; display: flex; align-items: center; justify-content: center; margin-bottom: 10px; border-radius: 4px;">No preview</div>`
                            }
                            <div style="margin-top: 10px;">
                                <small style="color: #666; display: block;">ID: ${design.designId.substring(0, 8)}...</small>
                                ${design.previewUrl ? `<button onclick="window.open('${design.previewUrl}', '_blank')" style="margin-top: 8px; padding: 4px 12px; background: #0073aa; color: white; border: none; border-radius: 3px; cursor: pointer; font-size: 12px;">View Full Size</button>` : ''}
                            </div>
                        </div>
                    `;
                });
                html += '</div>';
            } else {
                // Try to load designs by ID from backend if not in sessionStorage
                html += '<p>Loading designs...</p>';
                // For now just show IDs
                html += '<div style="margin-top: 20px;">';
                designIds.forEach((id, index) => {
                    html += `<div style="padding: 10px; background: #f0f0f0; margin-bottom: 10px; border-radius: 5px;">Design ${index + 1}: ${id}</div>`;
                });
                html += '</div>';
            }

            content.innerHTML = html;
            modal.appendChild(content);
            document.body.appendChild(modal);

            // Close on overlay click
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    modal.remove();
                }
            });
        }
        
        // Attach click handlers to all customize buttons
        document.addEventListener('DOMContentLoaded', function() {
            // Removed unnecessary cart button JavaScript - handled by PHP now

            document.addEventListener('click', function(e) {
                if (e.target.classList.contains('fz-customize-button')) {
                    e.preventDefault();
                    const url = e.target.getAttribute('data-customize-url');
                    const productName = e.target.getAttribute('data-product-name');
                    openFZModal(url, productName);
                }

                // Handle View All Designs link in cart
                if (e.target.classList.contains('fz-view-cart-designs') ||
                    e.target.closest('.fz-view-cart-designs')) {
                    e.preventDefault();
                    const target = e.target.classList.contains('fz-view-cart-designs') ?
                                   e.target : e.target.closest('.fz-view-cart-designs');
                    const designIds = target.getAttribute('data-design-ids');
                    if (designIds) {
                        showCartDesignsModal(designIds.split(','));
                    }
                }
            });
        });
        
        // Helper function to get current WooCommerce product ID
        function getCurrentWCProductId() {
            // Check if we're on a single product page
            const productForm = document.querySelector('form.cart');
            if (productForm) {
                // Try to get from add-to-cart button
                const addToCartBtn = productForm.querySelector('[name="add-to-cart"]');
                if (addToCartBtn) {
                    return addToCartBtn.value;
                }
                // Try to get from variation form
                const variationInput = productForm.querySelector('input[name="product_id"]');
                if (variationInput) {
                    return variationInput.value;
                }
            }
            
            // Try to get from customize button data attribute
            const customizeBtn = document.querySelector('[data-wc-product-id]');
            if (customizeBtn) {
                return customizeBtn.getAttribute('data-wc-product-id');
            }
            
            return null;
        }
        
        // Listen for messages from iframe
        window.addEventListener('message', function(e) {
            // Handle close modal message
            if (e.data && e.data.action === 'closeFZModal') {
                closeFZModal();
                return;
            }
            
            // Handle messages from customize iframe
            if (e.data && e.data.type === 'customization_saved') {
                console.log('Customization saved:', e.data);
                
                const productId = e.data.productId || e.data.wc_product_id || '';
                const designId = e.data.designId || e.data.customizationId || '';
                const wcProductId = e.data.customization?.wc_product_id || e.data.wc_product_id || getCurrentWCProductId();
                
                // Try to get preview URL from multiple possible locations
                let previewUrl = e.data.previewUrl || 
                                 e.data.preview_url || 
                                 e.data.customization?.preview_image_url || 
                                 e.data.customization?.preview_url || 
                                 '';
                
                console.log('Design details:', { productId, designId, previewUrl, wcProductId });
                
                if (designId && productId) {
                    // Store design info in multiple ways for better retrieval
                    sessionStorage.setItem('fz_design_' + productId, JSON.stringify({
                        designId: designId,
                        previewUrl: previewUrl,
                        productId: productId
                    }));
                    
                    // Also store by WC product ID if available
                    if (wcProductId) {
                        sessionStorage.setItem('fz_design_wc_' + wcProductId, JSON.stringify({
                            designId: designId,
                            previewUrl: previewUrl,
                            productId: productId
                        }));
                    }
                    
                    // Save to WooCommerce backend
                    if (window.fz_ajax_url) {
                        fetch(window.fz_ajax_url, {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/x-www-form-urlencoded',
                            },
                            body: new URLSearchParams({
                                action: 'fz_save_design',
                                design_id: designId,
                                product_id: productId,
                                preview_url: previewUrl,
                                preview_image_url: previewUrl, // Include both for compatibility
                                wc_product_id: productId,
                                nonce: window.fz_nonce || ''
                            })
                        }).then(response => response.json())
                        .then(data => {
                            console.log('Design saved to WooCommerce:', data);
                            // Update session storage with server response - support multiple designs
                            if (data.success && data.preview_url) {
                                // Store the current design (for backwards compatibility)
                                sessionStorage.setItem('fz_design_' + productId, JSON.stringify({
                                    designId: designId,
                                    previewUrl: data.preview_url,
                                    productId: productId
                                }));
                                
                                // Also store in multiple designs array
                                let allDesigns = JSON.parse(sessionStorage.getItem('fz_all_designs') || '{}');
                                if (!allDesigns[productId]) {
                                    allDesigns[productId] = [];
                                }
                                // Add new design if not already in list
                                const existingIndex = allDesigns[productId].findIndex(d => d.designId === designId);
                                if (existingIndex >= 0) {
                                    allDesigns[productId][existingIndex] = {
                                        designId: designId,
                                        previewUrl: data.preview_url,
                                        timestamp: Date.now()
                                    };
                                } else {
                                    allDesigns[productId].push({
                                        designId: designId,
                                        previewUrl: data.preview_url,
                                        timestamp: Date.now()
                                    });
                                }
                                sessionStorage.setItem('fz_all_designs', JSON.stringify(allDesigns));
                            }
                        });
                    }
                    
                    closeFZModal();

                    // Trigger event for multi-design manager to handle
                    if (window.fzDesignManager) {
                        window.fzDesignManager.handleMessage({
                            data: {
                                type: 'customization_saved',
                                designId: designId,
                                customization: {
                                    id: designId,
                                    preview_image_url: previewUrl
                                },
                                action: e.data.action || 'new',
                                previewUrl: previewUrl
                            }
                        });
                    }
                } else {
                    closeFZModal();
                }
            }
        });
        
        // Check for saved designs on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Check sessionStorage for saved designs
            const sections = document.querySelectorAll('.fz-customizer-section');
            sections.forEach(section => {
                const wcProductId = section.getAttribute('data-product-id');
                const fzProductId = section.getAttribute('data-fz-product-id');
                
                // Check multiple storage keys
                let savedDesign = null;
                const keys = [
                    'fz_design_' + fzProductId,
                    'fz_design_wc_' + wcProductId,
                    'fz_design_' + wcProductId
                ];
                
                for (const key of keys) {
                    const stored = sessionStorage.getItem(key);
                    if (stored) {
                        try {
                            savedDesign = JSON.parse(stored);
                            break;
                        } catch (e) {}
                    }
                }
                
                if (savedDesign && savedDesign.designId && window.fzDesignManager) {
                    // Add the saved design to the manager
                    window.fzDesignManager.handleMessage({
                        data: {
                            type: 'customization_saved',
                            designId: savedDesign.designId,
                            customization: {
                                id: savedDesign.designId,
                                preview_image_url: savedDesign.previewUrl
                            },
                            action: 'new',
                            previewUrl: savedDesign.previewUrl
                        }
                    });
                }
            });
            
            // Handle action buttons
            document.addEventListener('click', function(e) {
                const previewDiv = e.target.closest('.fz-saved-design-preview');
                if (!previewDiv) return;
                
                const designId = previewDiv.getAttribute('data-design-id');
                const previewUrl = previewDiv.getAttribute('data-preview-url');
                const productId = previewDiv.getAttribute('data-product-id');
                const wcProductId = previewDiv.getAttribute('data-wc-product-id');
                
                if (e.target.classList.contains('fz-edit-design')) {
                    // Get the customize URL from the main button
                    const customizeBtn = previewDiv.parentElement.querySelector('.fz-customize-button');
                    if (customizeBtn && designId) {
                        const url = customizeBtn.getAttribute('data-customize-url');
                        // Add design ID to URL for editing (make sure it's just the ID, not an object)
                        const editUrl = url + '&designId=' + encodeURIComponent(designId);
                        console.log('Opening editor with designId:', designId);
                        openFZModal(editUrl);
                    }
                } else if (e.target.classList.contains('fz-view-design')) {
                    // Open preview in new window
                    if (previewUrl) {
                        window.open(previewUrl, '_blank');
                    } else {
                        alert('Preview image is not available yet. Please try again in a moment.');
                    }
                } else if (e.target.classList.contains('fz-add-more-design')) {
                    // Open customizer for another design
                    const customizeBtn = previewDiv.parentElement.querySelector('.fz-customize-button');
                    if (customizeBtn) {
                        const url = customizeBtn.getAttribute('data-customize-url');
                        console.log('Opening editor for new design');
                        openFZModal(url);
                    }
                }
            });
            
            // Inject design ID into the WooCommerce add to cart form
            document.querySelectorAll('.fz-customizer-section').forEach(section => {
                const previewDiv = section.querySelector('.fz-saved-design-preview');
                const wcProductId = section.getAttribute('data-product-id');
                
                if (previewDiv && previewDiv.style.display !== 'none') {
                    const designId = previewDiv.getAttribute('data-design-id');
                    
                    if (designId && wcProductId) {
                        // Find the add to cart form for this product
                        const addToCartForm = document.querySelector('form.cart');
                        if (addToCartForm) {
                            // Add hidden input with design ID if not already present
                            let designInput = addToCartForm.querySelector('input[name="fz_design_id"]');
                            if (!designInput) {
                                designInput = document.createElement('input');
                                designInput.type = 'hidden';
                                designInput.name = 'fz_design_id';
                                addToCartForm.appendChild(designInput);
                            }
                            designInput.value = designId;
                            
                            // Also store in sessionStorage for multiple designs support
                            let designs = JSON.parse(sessionStorage.getItem('fz_cart_designs') || '{}');
                            if (!designs[wcProductId]) {
                                designs[wcProductId] = [];
                            }
                            if (!designs[wcProductId].includes(designId)) {
                                designs[wcProductId].push(designId);
                            }
                            sessionStorage.setItem('fz_cart_designs', JSON.stringify(designs));
                        }
                    }
                }
            });
        });
        
        // Function to show design saved dialog (keeping for backward compatibility)
        function showDesignSavedDialog(productId, designId, previewUrl) {
            const dialog = document.createElement('div');
            dialog.className = 'fz-design-saved-dialog';
            dialog.innerHTML = `
                <div class="fz-dialog-content">
                    <h3>✅ Design Saved Successfully!</h3>
                    ${previewUrl ? `<img src="${previewUrl}" alt="Design Preview" style="max-width: 300px; margin: 20px 0; border: 1px solid #ddd; border-radius: 8px;">` : ''}
                    <div class="fz-dialog-actions">
                        <button onclick="addCustomizedToCart('${productId}', '${designId}')" class="button alt">
                            🛒 Add to Cart
                        </button>
                        <button onclick="downloadDesign('${previewUrl || ''}')" class="button">
                            📥 Download Design
                        </button>
                        <button onclick="closeDesignDialog()" class="button">
                            Continue Shopping
                        </button>
                    </div>
                </div>
            `;
            document.body.appendChild(dialog);
            
            // Add styles for dialog
            const style = document.createElement('style');
            style.textContent = `
                .fz-design-saved-dialog {
                    position: fixed;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    background: white;
                    padding: 30px;
                    border-radius: 10px;
                    box-shadow: 0 10px 40px rgba(0,0,0,0.2);
                    z-index: 999999;
                    max-width: 500px;
                    text-align: center;
                }
                .fz-dialog-actions {
                    display: flex;
                    gap: 10px;
                    justify-content: center;
                    margin-top: 20px;
                }
                .fz-dialog-actions button {
                    flex: 1;
                }
            `;
            document.head.appendChild(style);
        }
        
        // Add to cart silently (without popup)
        function addCustomizedToCartSilently(productId, designId, previewUrl) {
            // Get the WooCommerce product ID from the page
            const wcProductId = document.querySelector('[data-product-id]')?.dataset.productId || 
                               new URLSearchParams(window.location.search).get('wc_product_id') ||
                               productId;
            
            // Store design info for the product
            sessionStorage.setItem('fz_design_preview_' + wcProductId, previewUrl);
            
            // Add to cart using WooCommerce AJAX
            if (window.jQuery && window.jQuery.fn.trigger) {
                // Find the add to cart button for this product
                const addToCartBtn = document.querySelector('.single_add_to_cart_button');
                if (addToCartBtn) {
                    // Add design data to form
                    const form = addToCartBtn.closest('form');
                    if (form) {
                        // Add hidden input with design ID
                        let designInput = form.querySelector('input[name="fz_design_id"]');
                        if (!designInput) {
                            designInput = document.createElement('input');
                            designInput.type = 'hidden';
                            designInput.name = 'fz_design_id';
                            form.appendChild(designInput);
                        }
                        designInput.value = designId;
                        
                        // Submit the form
                        form.submit();
                    }
                } else {
                    // Fallback: redirect with parameters
                    window.location.href = '?add-to-cart=' + wcProductId + '&fz_design_id=' + designId;
                }
            } else {
                // Direct redirect with parameters
                window.location.href = '?add-to-cart=' + wcProductId + '&fz_design_id=' + designId;
            }
        }
        
        // Helper functions
        window.addCustomizedToCart = function(productId, designId) {
            if (window.fz_ajax_url) {
                fetch(window.fz_ajax_url, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'fz_add_customized_to_cart',
                        product_id: productId,
                        design_id: designId,
                        quantity: 1,
                        nonce: window.fz_nonce || ''
                    })
                }).then(response => response.json())
                .then(data => {
                    if (data.success) {
                        window.location.href = data.data.cart_url || '/cart';
                    } else {
                        alert('Failed to add to cart. Please try again.');
                    }
                });
            } else {
                // Fallback: add to cart with design parameter
                window.location.href = '?add-to-cart=' + productId + '&fz_design_id=' + designId;
            }
        };
        
        window.downloadDesign = function(previewUrl) {
            if (previewUrl) {
                const link = document.createElement('a');
                link.href = previewUrl;
                link.download = 'custom-design.png';
                link.click();
            }
        };
        
        window.closeDesignDialog = function() {
            const dialog = document.querySelector('.fz-design-saved-dialog');
            if (dialog) {
                dialog.remove();
            }
        };
    </script>
    <?php
});

// Handle webhook from FZ Dashboard
add_action('rest_api_init', function() {
    register_rest_route('fz-customizer/v1', '/webhook', [
        'methods' => 'POST',
        'callback' => 'fz_handle_webhook',
        'permission_callback' => 'fz_verify_webhook'
    ]);
});

function fz_verify_webhook(WP_REST_Request $request) {
    $auth_header = $request->get_header('Authorization');
    $stored_token = get_option('fz_auth_token');
    
    if (!$auth_header || !$stored_token) {
        return false;
    }
    
    $provided_token = str_replace('Bearer ', '', $auth_header);
    return $provided_token === $stored_token;
}

function fz_handle_webhook(WP_REST_Request $request) {
    $data = $request->get_json_params();
    
    // Handle different webhook types
    switch ($data['type']) {
        case 'order_created':
            // Handle order creation
            break;
        case 'design_saved':
            // Handle design saved
            break;
    }
    
    return new WP_REST_Response(['success' => true], 200);
}

// Include API endpoints
require_once FZ_CUSTOMIZER_PLUGIN_DIR . 'includes/fz-customizer-api.php';