/**
 * Multi-Design Manager for WooCommerce Product Page
 */
(function($) {
    'use strict';

    class MultiDesignManager {
        constructor() {
            this.designs = [];
            this.currentDesignId = null;
            this.productId = fz_customizer_params?.product_id;
            this.processingMessage = false; // Prevent duplicate processing
            this.init();
        }

        init() {
            // Load saved designs from sessionStorage
            this.loadSavedDesigns();

            // Listen for messages from iframe (this is the main way designs are saved)
            window.addEventListener('message', this.handleIframeMessage.bind(this));

            // Handle Add Another Design button
            $(document).on('click', '.fz-add-another-design', this.handleAddAnother.bind(this));

            // Handle Edit Design button
            $(document).on('click', '.fz-edit-design', this.handleEditDesign.bind(this));

            // Handle Delete Design button
            $(document).on('click', '.fz-delete-design', this.handleDeleteDesign.bind(this));

            // Handle View Full Size button
            $(document).on('click', '.fz-view-design', this.handleViewDesign.bind(this));

            // Handle clicking on design thumbnail
            $(document).on('click', '.fz-design-thumbnail', this.handleThumbnailClick.bind(this));

            // Handle quantity change in preview modal
            $(document).on('change', '.fz-design-quantity', this.handleQuantityChange.bind(this));

            // Initial render if we have saved designs
            if (this.designs.length > 0) {
                this.renderDesigns();
            }
        }

        loadSavedDesigns() {
            // Load designs from sessionStorage
            const allDesigns = JSON.parse(sessionStorage.getItem('fz_all_designs') || '{}');
            const productDesigns = allDesigns[this.productId] || [];

            // Convert to our format, ensuring each design has a quantity
            this.designs = productDesigns.map(d => ({
                id: d.designId,
                previewUrl: d.previewUrl,
                quantity: d.quantity || 1,
                data: d
            }));

            // Also check for single design (backwards compatibility)
            const singleDesign = sessionStorage.getItem('fz_design_' + this.productId);
            if (singleDesign && this.designs.length === 0) {
                try {
                    const design = JSON.parse(singleDesign);
                    if (design.designId) {
                        this.designs.push({
                            id: design.designId,
                            previewUrl: design.previewUrl,
                            quantity: 1,
                            data: design
                        });
                    }
                } catch (e) {}
            }
        }

        saveDesignsToStorage() {
            // Save to sessionStorage
            let allDesigns = JSON.parse(sessionStorage.getItem('fz_all_designs') || '{}');
            allDesigns[this.productId] = this.designs.map(d => ({
                designId: d.id,
                previewUrl: d.previewUrl,
                quantity: d.quantity || 1,
                timestamp: Date.now()
            }));
            sessionStorage.setItem('fz_all_designs', JSON.stringify(allDesigns));
        }

        // Handle messages from iframe only
        handleIframeMessage(event) {
            if (!event.data || !event.data.type) return;

            // Only process if it's from the iframe (not from our own calls)
            if (event.source === window) return;

            switch (event.data.type) {
                case 'customization_saved':
                    this.handleDesignSaved(event.data);
                    break;
                case 'design_deleted':
                    this.handleDesignDeleted(event.data);
                    break;
                case 'close_editor':
                    this.closeEditor();
                    break;
            }
        }

        // Direct call handler (for backwards compatibility)
        handleMessage(data) {
            // Prevent processing if we're already processing from iframe
            if (this.processingMessage) return;

            // Mark as processing
            this.processingMessage = true;

            // Only handle if we don't already have this design
            if (data.data && data.data.type === 'customization_saved') {
                const designId = data.data.designId;
                if (!this.designs.find(d => d.id === designId)) {
                    this.handleDesignSaved(data.data);
                }
            }

            // Reset flag
            setTimeout(() => {
                this.processingMessage = false;
            }, 100);
        }

        handleDesignSaved(data) {
            const { designId, customization, action, previewUrl } = data;

            // Check if design already exists to prevent duplicates
            const existingIndex = this.designs.findIndex(d => d.id === designId);

            if (action === 'new' || action === 'add_another' || existingIndex === -1) {
                // Only add if it doesn't exist
                if (existingIndex === -1) {
                    this.designs.push({
                        id: designId,
                        previewUrl: previewUrl || customization?.preview_image_url,
                        quantity: 1,
                        data: customization
                    });
                }
            } else {
                // Update existing design
                this.designs[existingIndex] = {
                    ...this.designs[existingIndex],
                    previewUrl: previewUrl || customization?.preview_image_url,
                    data: customization
                };
            }

            this.saveDesignsToStorage();
            this.renderDesigns();
            this.showSuccessMessage();
        }

        handleDesignDeleted(data) {
            const { designId } = data;
            this.designs = this.designs.filter(d => d.id !== designId);
            this.saveDesignsToStorage();
            this.renderDesigns();
        }

        renderDesigns() {
            const container = $('#fz-designs-container');
            if (!container.length) {
                return;
            }

            // If no designs, empty the container and hide it
            if (this.designs.length === 0) {
                container.empty();
                $('.fz-add-another-wrapper-outside').remove();
                return;
            }

            let html = '<div class="fz-saved-designs">';
            html += '<div class="fz-designs-header">Your Custom Designs</div>';
            html += '<div class="fz-designs-grid">';

            this.designs.forEach((design, index) => {
                const quantity = design.quantity || 1;
                html += `
                    <div class="fz-design-item" data-design-id="${design.id}">
                        <div class="fz-design-thumbnail" style="cursor: pointer;" data-design-id="${design.id}">
                            ${design.previewUrl ?
                                `<img src="${design.previewUrl}" alt="Design ${index + 1}">` :
                                `<div class="fz-design-placeholder">Design ${index + 1}</div>`
                            }
                            <div class="fz-design-number">${index + 1}</div>
                            ${quantity > 1 ? `<div class="fz-design-quantity-badge" style="position: absolute; bottom: 4px; right: 4px; background: #ff6b6b; color: white; padding: 2px 6px; border-radius: 10px; font-size: 11px; font-weight: bold;">×${quantity}</div>` : ''}
                        </div>
                        <div class="fz-design-actions">
                            <button class="fz-edit-design" data-design-id="${design.id}" title="Edit" style="font-size: 14px;">
                                ✏️
                            </button>
                            <button class="fz-view-design" data-design-id="${design.id}" title="Preview" style="font-size: 14px;">
                                👁️
                            </button>
                            <button class="fz-delete-design" data-design-id="${design.id}" title="Delete" style="font-size: 14px;">
                                🗑️
                            </button>
                        </div>
                    </div>
                `;
            });

            html += '</div></div>';
            container.html(html);

            // Add the "Add Another Design" button outside the container
            this.addExternalAddButton();

            this.updateQuantity();
            $(document).trigger('fz:designs-updated');
        }

        addExternalAddButton() {
            // Remove any existing external button
            $('.fz-add-another-wrapper-outside').remove();

            // Add button after the container
            const container = $('#fz-designs-container');
            if (container.length && this.designs.length > 0) {
                const buttonHtml = `
                    <div class="fz-add-another-wrapper-outside" style="margin-top: 15px;">
                        <button class="button fz-add-another-design" style="background: #46b450; color: white; border: none; padding: 10px 20px; font-weight: 500;">
                            <span style="font-size: 16px; margin-right: 5px;">+</span>
                            Add Another Design
                        </button>
                    </div>
                `;
                container.after(buttonHtml);
            }
        }

        handleThumbnailClick(e) {
            e.stopPropagation();
            const designId = $(e.currentTarget).data('design-id');
            const design = this.designs.find(d => d.id === designId);
            if (design) {
                this.showPreviewModal(design);
            }
        }

        handleAddAnother() {
            // Get the customize button URL
            const customizeBtn = $('.fz-customize-button');
            if (customizeBtn.length) {
                let url = customizeBtn.data('customize-url');
                const productName = customizeBtn.data('product-name');

                // Add action parameter for new design
                if (url) {
                    // Clear any existing savedDesignId to force new design
                    url = url.replace(/&savedDesignId=[^&]*/, '');
                    url += (url.includes('?') ? '&' : '?') + 'action=add_another';
                    // Use the existing openFZModal function
                    if (typeof window.openFZModal === 'function') {
                        window.openFZModal(url, productName);
                    }
                }
            }
        }

        handleEditDesign(e) {
            e.stopPropagation();
            const designId = $(e.currentTarget).data('design-id');
            const customizeBtn = $('.fz-customize-button');

            if (customizeBtn.length) {
                let url = customizeBtn.data('customize-url');
                const productName = customizeBtn.data('product-name');

                // Add design ID for editing
                if (url) {
                    url += (url.includes('?') ? '&' : '?') + 'savedDesignId=' + designId + '&action=edit';
                    if (typeof window.openFZModal === 'function') {
                        window.openFZModal(url, productName);
                    }
                }
            }
        }

        handleDeleteDesign(e) {
            e.stopPropagation();
            const designId = $(e.currentTarget).data('design-id');
            if (confirm('Are you sure you want to delete this design?')) {
                this.designs = this.designs.filter(d => d.id !== designId);
                this.saveDesignsToStorage();
                this.renderDesigns();
                this.updateQuantity();
            }
        }

        handleViewDesign(e) {
            e.stopPropagation();
            const designId = $(e.currentTarget).data('design-id');
            const design = this.designs.find(d => d.id === designId);
            if (design) {
                this.showPreviewModal(design);
            }
        }

        showPreviewModal(design) {
            const designIndex = this.designs.findIndex(d => d.id === design.id);
            const quantity = design.quantity || 1;

            const modal = $(`
                <div class="fz-preview-modal">
                    <div class="fz-preview-overlay"></div>
                    <div class="fz-preview-content" style="max-width: 600px; padding: 20px;">
                        <button class="fz-preview-close">&times;</button>
                        <h3 style="margin-top: 0; margin-bottom: 15px;">Design ${designIndex + 1} Preview</h3>
                        ${design.previewUrl ?
                            `<img src="${design.previewUrl}" alt="Design Preview" style="max-width: 100%; margin-bottom: 20px;">` :
                            `<div style="background: #f0f0f0; padding: 40px; text-align: center; margin-bottom: 20px;">No preview available</div>`
                        }
                        <div style="background: #f8f9fa; padding: 15px; border-radius: 5px;">
                            <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 15px;">
                                <label style="font-weight: 600;">Quantity:</label>
                                <div style="display: flex; align-items: center; gap: 10px;">
                                    <button class="fz-qty-decrease" data-design-id="${design.id}" style="width: 30px; height: 30px; border: 1px solid #ddd; background: white; cursor: pointer;">-</button>
                                    <input type="number" class="fz-design-quantity" data-design-id="${design.id}" value="${quantity}" min="1" style="width: 60px; text-align: center; border: 1px solid #ddd; padding: 4px;">
                                    <button class="fz-qty-increase" data-design-id="${design.id}" style="width: 30px; height: 30px; border: 1px solid #ddd; background: white; cursor: pointer;">+</button>
                                </div>
                            </div>
                            <div style="display: flex; gap: 10px;">
                                <button class="button fz-edit-design" data-design-id="${design.id}" style="flex: 1; background: #0073aa; color: white; border: none; padding: 10px;">
                                    Edit Design
                                </button>
                                <button class="button fz-delete-design" data-design-id="${design.id}" style="flex: 1; background: #dc3232; color: white; border: none; padding: 10px;">
                                    Delete Design
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `);

            $('body').append(modal);

            // Handle close
            modal.find('.fz-preview-close, .fz-preview-overlay').on('click', function() {
                modal.remove();
            });

            // Handle quantity buttons
            modal.find('.fz-qty-decrease').on('click', (e) => {
                const input = modal.find('.fz-design-quantity');
                const val = parseInt(input.val()) || 1;
                if (val > 1) {
                    input.val(val - 1).trigger('change');
                }
            });

            modal.find('.fz-qty-increase').on('click', (e) => {
                const input = modal.find('.fz-design-quantity');
                const val = parseInt(input.val()) || 1;
                input.val(val + 1).trigger('change');
            });

            // Handle actions from modal
            modal.find('.fz-edit-design').on('click', (e) => {
                modal.remove();
                this.handleEditDesign(e);
            });

            modal.find('.fz-delete-design').on('click', (e) => {
                modal.remove();
                this.handleDeleteDesign(e);
            });
        }

        handleQuantityChange(e) {
            const designId = $(e.target).data('design-id');
            const quantity = parseInt($(e.target).val()) || 1;

            const design = this.designs.find(d => d.id === designId);
            if (design) {
                design.quantity = Math.max(1, quantity);
                this.saveDesignsToStorage();
                this.renderDesigns();
                this.updateQuantity();
            }
        }

        closeEditor() {
            if (typeof window.closeFZModal === 'function') {
                window.closeFZModal();
            }
        }

        showSuccessMessage() {
            let successMsg = $('.fz-design-saved-message');
            if (!successMsg.length) {
                successMsg = $('<div class="fz-design-saved-message woocommerce-message" style="background: #d4edda; color: #155724; padding: 12px 20px; border: 1px solid #c3e6cb; border-radius: 4px; margin: 10px 0;">✓ Design Saved Successfully!</div>');
                $('#fz-designs-container').before(successMsg);
            }

            successMsg.fadeIn().delay(3000).fadeOut();
        }

        updateQuantity() {
            // Calculate total quantity across all designs
            const totalQty = this.designs.reduce((sum, design) => sum + (design.quantity || 1), 0);

            // Update quantity input
            const qtyInput = $('input.qty');
            if (qtyInput.length && totalQty > 0) {
                qtyInput.val(totalQty);
                qtyInput.trigger('change');
            }

            // Update the original Add to Cart button text
            const addToCartBtn = $('.single_add_to_cart_button');
            if (addToCartBtn.length) {
                if (totalQty > 0) {
                    if (!addToCartBtn.data('original-text')) {
                        addToCartBtn.data('original-text', addToCartBtn.text());
                    }
                    const designCount = this.designs.length;
                    addToCartBtn.text(`Add to cart (${totalQty} item${totalQty !== 1 ? 's' : ''}, ${designCount} design${designCount !== 1 ? 's' : ''})`);
                } else {
                    const originalText = addToCartBtn.data('original-text');
                    if (originalText) {
                        addToCartBtn.text(originalText);
                    }
                }
            }

            // Store designs data for cart submission
            let cartForm = $('form.cart');
            if (cartForm.length) {
                // Remove old hidden inputs
                cartForm.find('input[name^="fz_designs"]').remove();

                if (this.designs.length > 0) {
                    // Add hidden inputs for each design with quantity
                    this.designs.forEach((design, index) => {
                        const qty = design.quantity || 1;
                        for (let i = 0; i < qty; i++) {
                            cartForm.append(`<input type="hidden" name="fz_designs[]" value="${design.id}">`);
                        }
                    });
                }
            }
        }
    }

    // Initialize on document ready
    $(document).ready(function() {
        if ($('.fz-customize-button').length) {
            window.fzDesignManager = new MultiDesignManager();
        }
    });

})(jQuery);